@extends('site.base')
@section('titulo', 'Crie sua conta ' . env('EMPRESA'))
@section('url', route('site.cadastro'))
@section('description', 'Crie sua conta ' . env('EMPRESA'))

@section('content')
    <div class="relative w-full false">

        {{-- FORM --}}
        <div class="pt-[64px]">
            <div class="relative w-[95%] max-h-[100%] md:w-4/6 lg:w-3/6 xl:w-[40%] 4xl:w-[25%] my-6 mx-auto">
                <div class="overflow-auto rounded-2xl shadow-lg flex flex-col w-full h-full bg-white">
                    <div class="h-full p-6">
                        <h1 class="text-primary-green font-bold text-2xl mb-3 mt-6">Criar Conta</h1>

                        <form id="registerForm" action="{{ route('register.new') }}" method="POST" novalidate>
                            @csrf

                            {{-- Nome --}}
                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">Nome completo</label>
                                <input type="text" id="nome" name="nome" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" placeholder="Insira seu nome completo"
                                    value="{{ old('nome') }}" required>
                                <small id="nomeHelp" class="text-danger"></small>
                            </div>

                            {{-- RG / CPF --}}
                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">RG</label>
                                    <input type="text" id="rg" name="rg" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" placeholder="Insira seu RG"
                                        value="{{ old('rg') }}" required>
                                    <small id="rgHelp" class="text-danger"></small>
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">CPF</label>
                                    <input type="text" id="cpf" name="cpf" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" placeholder="000.000.000-00"
                                        value="{{ old('cpf') }}" required>
                                    <small id="cpfHelp" class="text-danger"></small>
                                </div>
                            </div>

                            {{-- Celular / Nascimento --}}
                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Celular (WhatsApp)</label>
                                    <input type="tel" id="celular" name="celular" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" placeholder="(00) 00000-0000"
                                        value="{{ old('celular') }}" required>
                                    <small id="celularHelp" class="text-danger"></small>
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Data de Nascimento</label>
                                    <input type="date" id="nascimento" name="nascimento" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" value="{{ old('nascimento') }}"
                                        required>
                                    <small id="nascimentoHelp" class="text-danger"></small>
                                </div>
                            </div>

                            {{-- Sexo --}}
                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">Sexo</label>
                                <div class="flex gap-4 mt-2 toggle-group-like">
                                    <label><input type="radio" name="sexo" value="Masculino" {{ old('sexo', 'Masculino') == 'Masculino' ? 'checked' : '' }}> Masculino</label>
                                    <label><input type="radio" name="sexo" value="Feminino" {{ old('sexo') == 'Feminino' ? 'checked' : '' }}> Feminino</label>
                                </div>
                                <small id="sexoHelp" class="text-danger"></small>
                            </div>

                            {{-- Endereço --}}
                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">CEP</label>
                                <input type="text" id="cep" name="cep" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" placeholder="00000-000"
                                    value="{{ old('cep') }}" required>
                                <small id="cepHelp" class="text-danger"></small>
                            </div>

                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">Endereço</label>
                                <input type="text" id="endereco" name="endereco" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" placeholder="Rua/Avenida"
                                    value="{{ old('endereco') }}" required>
                                <small id="enderecoHelp" class="text-danger"></small>
                            </div>

                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Número</label>
                                    <input type="text" id="numero" name="numero" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" value="{{ old('numero') }}" required>
                                    <small id="numeroHelp" class="text-danger"></small>
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Bairro</label>
                                    <input type="text" id="bairro" name="bairro" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" value="{{ old('bairro') }}"
                                        required>
                                    <small id="bairroHelp" class="text-danger"></small>
                                </div>
                            </div>

                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Cidade</label>
                                    <input type="text" id="cidade" name="cidade" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" value="{{ old('cidade') }}"
                                        required>
                                    <small id="cidadeHelp" class="text-danger"></small>
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Estado (UF)</label>
                                    <select id="estado" name="estado" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" maxlength="2" style="text-transform:uppercase"
                                        required>
                                        @foreach ($Estados as $estado)
                                            <option value="{{ $estado->uf }}">{{ $estado->nome }}</option>
                                        @endforeach
                                    </select>
                                    <small id="estadoHelp" class="text-danger"></small>
                                </div>
                            </div>

                            {{-- Email / Senha --}}
                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">E-mail</label>
                                <input type="email" id="email" name="email" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" placeholder="seu@email.com"
                                    value="{{ old('email') }}" required>
                                <small id="emailHelp" class="text-danger"></small>
                            </div>

                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Senha</label>
                                    <input type="password" id="password" name="password" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60" minlength="8" required>
                                    <small id="passHelp" class="text-danger"></small>
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Confirmar Senha</label>
                                    <input type="password" id="password_confirmation" name="password_confirmation" class="bg-green-light w-full h-11 px-4 rounded-full text-tertiary-green text-base focus:outline-none focus:bg-green-light/60"
                                        minlength="8" required>
                                    <small id="passwordConfirmationHelp" class="text-danger"></small>
                                </div>
                            </div>

                            {{-- Botões --}}
                            <div class="flex flex-col md:flex-row gap-4 justify-center mt-6">
                                <button type="reset" id="btnReset" class="bg-green-button w-full hover:bg-green-hover rounded-full px-4 py-[0.6rem]">
                                    <span class="text-center text-primary-green">Limpar</span>
                                </button>
                                <button type="submit" id="btnSubmit" class="bg-primary-green w-full hover:bg-primary-green/90 rounded-full px-4 py-[0.6rem]">
                                    <span class="text-center text-white">Cadastrar</span>
                                </button>
                            </div>
                        </form>

                        @if ($errors->any())
                            <div class="alert alert-danger mt-3">
                                <strong>Ops!</strong> Verifique os campos destacados e tente novamente.
                            </div>
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('css')
    <style>
        /* simples para feedback visual imediato (mantive suas classes) */
        .is-valid {
            border: 2px solid #28a745 !important;
        }

        .is-invalid {
            border: 2px solid #dc3545 !important;
        }

        /* pequenos ajustes para textos de ajuda */
        small.text-danger {
            color: #dc3545;
            display: block;
            margin-top: 6px;
            min-height: 18px;
        }
    </style>
@endsection

@section('js')
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>

    <script>
        (function() {
            const CSRF = $('meta[name="csrf-token"]').attr('content');

            /* ===== Helpers ===== */
            function onlyDigits(v) {
                return (v || '').replace(/\D/g, '');
            }

            function markInvalid($el, helpSel, msg) {
                $el.addClass('is-invalid').removeClass('is-valid');
                if (helpSel) $(helpSel).text(msg || 'Campo obrigatório');
            }

            function markValid($el, helpSel, msg) {
                $el.addClass('is-valid').removeClass('is-invalid');
                if (helpSel) $(helpSel).text(msg || '');
            }

            function clearInvalid($el, helpSel) {
                $el.removeClass('is-invalid is-valid');
                if (helpSel) $(helpSel).text('');
            }

            function isEmpty($el) {
                return !($el.val() || '').toString().trim();
            }

            /* ===== Máscaras ===== */
            $('#cpf').on('input', function() {
                let v = onlyDigits($(this).val()).slice(0, 11);
                v = v.replace(/(\d{3})(\d)/, '$1.$2');
                v = v.replace(/(\d{3})(\d)/, '$1.$2');
                v = v.replace(/(\d{3})(\d{1,2})$/, '$1-$2');
                $(this).val(v);
            });

            $('#celular').on('input', function() {
                let v = onlyDigits($(this).val()).slice(0, 11);
                if (v.length >= 11) v = v.replace(/(\d{2})(\d{5})(\d{0,4})/, '($1) $2-$3');
                else if (v.length >= 10) v = v.replace(/(\d{2})(\d{4})(\d{0,4})/, '($1) $2-$3');
                else if (v.length > 2) v = v.replace(/(\d{2})(\d{0,5})/, '($1) $2');
                else if (v.length > 0) v = v.replace(/(\d{0,2})/, '($1');
                $(this).val(v);
                clearInvalid($('#celular'), '#celularHelp');
            });

            $('#cep').on('input', function() {
                let v = onlyDigits($(this).val()).slice(0, 8);
                if (v.length > 5) v = v.slice(0, 5) + '-' + v.slice(5);
                $(this).val(v);
            });

            $('#estado').on('input', function() {
                this.value = this.value.toUpperCase().slice(0, 2);
            });

            /* ===== Validações locais ===== */
            function isValidCPF(c) {
                c = onlyDigits(c);
                if (!c || c.length !== 11 || /^(\d)\1+$/.test(c)) return false;
                let s = 0,
                    r;
                for (let i = 1; i <= 9; i++) s += parseInt(c.substring(i - 1, i)) * (11 - i);
                r = (s * 10) % 11;
                if (r == 10 || r == 11) r = 0;
                if (r != parseInt(c.substring(9, 10))) return false;
                s = 0;
                for (let i = 1; i <= 10; i++) s += parseInt(c.substring(i - 1, i)) * (12 - i);
                r = (s * 10) % 11;
                if (r == 10 || r == 11) r = 0;
                return r == parseInt(c.substring(10, 11));
            }

            function isValidEmail(e) {
                return /^[^\s@]+@[^\s@]+\.[^\s@]{2,}$/.test((e || '').toLowerCase());
            }

            function isValidCell(full) {
                const d = onlyDigits(full);
                return d.length === 10 || d.length === 11;
            }

            function extractDDDAndNumber(full) {
                const d = onlyDigits(full);
                return d.length < 10 ? {
                    ddd: '',
                    numero: ''
                } : {
                    ddd: d.slice(0, 2),
                    numero: d.slice(2)
                };
            }

            /* ===== ViaCEP (jsonp) ===== */
            $('#cep').on('blur', function() {
                const numCep = onlyDigits($('#cep').val());
                if (numCep.length !== 8) {
                    markInvalid($('#cep'), '#cepHelp', 'CEP inválido (8 dígitos).');
                    return;
                }
                $('#cepHelp').text('Consultando CEP...');
                $.ajax({
                    url: 'https://viacep.com.br/ws/' + numCep + '/json/?callback=?',
                    dataType: 'jsonp',
                    success: function(d) {
                        if (d.erro) {
                            markInvalid($('#cep'), '#cepHelp', 'CEP não encontrado.');
                            return;
                        }
                        $('#cep').val(d.cep);
                        $('#endereco').val(d.logradouro || '');
                        $('#bairro').val(d.bairro || '');
                        $('#cidade').val(d.localidade || '');
                        $('#estado').val(d.uf || '');
                        markValid($('#cep'), '#cepHelp', '');
                    },
                    error: function() {
                        markInvalid($('#cep'), '#cepHelp', 'Erro ao consultar CEP.');
                    }
                });
            });

            /* ===== AJAX setup ===== */
            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': CSRF
                }
            });

            /* ===== Async validators (CPF / Celular / Email) - same logic as seu exemplo ===== */
            let lastCpfChecked = '',
                lastOutcomeCpf = '',
                cpfAjax = null;
            let lastPhoneChecked = '',
                lastOutcomePhone = '',
                phoneAjax = null;
            let lastEmailChecked = '',
                lastOutcomeEmail = '',
                emailAjax = null;

            const validarCpf = () => new Promise((resolve) => {
                const raw = $('#cpf').val() || '',
                    digits = onlyDigits(raw);
                if (digits.length < 11 || !isValidCPF(digits)) {
                    markInvalid($('#cpf'), '#cpfHelp', 'CPF inválido.');
                    return resolve(false);
                }
                if (digits === lastCpfChecked && lastOutcomeCpf) {
                    const ok = (lastOutcomeCpf === 'ok');
                    ok ? markValid($('#cpf'), '#cpfHelp') : markInvalid($('#cpf'), '#cpfHelp', 'CPF já cadastrado.');
                    return resolve(ok);
                }
                if (cpfAjax && cpfAjax.readyState !== 4) cpfAjax.abort();
                cpfAjax = $.post("{{ route('validar.cpf') }}", {
                        cpf: raw
                    })
                    .done(res => {
                        lastCpfChecked = digits;
                        if (res === 'cpf_error') {
                            lastOutcomeCpf = 'cpf_error';
                            markInvalid($('#cpf'), '#cpfHelp', 'CPF inválido.');
                            return resolve(false);
                        }
                        if (res === 'cpf_exist') {
                            lastOutcomeCpf = 'cpf_exist';
                            markInvalid($('#cpf'), '#cpfHelp', 'Este CPF já está cadastrado.');
                            return resolve(false);
                        }
                        lastOutcomeCpf = 'ok';
                        markValid($('#cpf'), '#cpfHelp', '');
                        resolve(true);
                    })
                    .fail(() => {
                        lastOutcomeCpf = '';
                        markInvalid($('#cpf'), '#cpfHelp', 'Não foi possível validar o CPF.');
                        resolve(false);
                    });
            });
            $('#cpf').on('blur', validarCpf);

            const validarCelular = () => new Promise((resolve) => {
                const full = $('#celular').val() || '';
                if (!isValidCell(full)) {
                    markInvalid($('#celular'), '#celularHelp', 'Informe DDD + número.');
                    return resolve(false);
                }
                const fullDigits = onlyDigits(full);
                if (fullDigits === lastPhoneChecked && lastOutcomePhone) {
                    const ok = (lastOutcomePhone === 'ok');
                    ok ? markValid($('#celular'), '#celularHelp') : markInvalid($('#celular'), '#celularHelp', 'Celular já cadastrado.');
                    return resolve(ok);
                }
                const {
                    ddd,
                    numero
                } = extractDDDAndNumber(full);
                if (!ddd || !numero) {
                    markInvalid($('#celular'), '#celularHelp', 'Informe DDD + número.');
                    return resolve(false);
                }
                if (phoneAjax && phoneAjax.readyState !== 4) phoneAjax.abort();
                phoneAjax = $.post("{{ route('validar.celular') }}", {
                        ddd,
                        celular: numero
                    })
                    .done(res => {
                        lastPhoneChecked = fullDigits;
                        if (res === 'celular_error') {
                            lastOutcomePhone = 'celular_error';
                            markInvalid($('#celular'), '#celularHelp', 'Número inválido.');
                            return resolve(false);
                        }
                        if (res === 'celular_exist') {
                            lastOutcomePhone = 'celular_exist';
                            markInvalid($('#celular'), '#celularHelp', 'Este celular já está cadastrado.');
                            return resolve(false);
                        }
                        lastOutcomePhone = 'ok';
                        markValid($('#celular'), '#celularHelp', '');
                        resolve(true);
                    })
                    .fail(() => {
                        lastOutcomePhone = '';
                        markInvalid($('#celular'), '#celularHelp', 'Não foi possível validar o celular.');
                        resolve(false);
                    });
            });
            $('#celular').on('blur', validarCelular);

            const validarEmail = () => new Promise((resolve) => {
                const email = ($('#email').val() || '').trim().toLowerCase();
                if (!isValidEmail(email)) {
                    markInvalid($('#email'), '#emailHelp', 'E-mail inválido.');
                    return resolve(false);
                }
                if (email === lastEmailChecked && lastOutcomeEmail) {
                    const ok = (lastOutcomeEmail === 'ok');
                    ok ? markValid($('#email'), '#emailHelp') : markInvalid($('#email'), '#emailHelp', 'E-mail já cadastrado.');
                    return resolve(ok);
                }
                if (emailAjax && emailAjax.readyState !== 4) emailAjax.abort();
                emailAjax = $.post("{{ route('validar.email') }}", {
                        email
                    })
                    .done(res => {
                        lastEmailChecked = email;
                        if (res === 'email_exist') {
                            lastOutcomeEmail = 'email_exist';
                            markInvalid($('#email'), '#emailHelp', 'Este e-mail já está cadastrado.');
                            return resolve(false);
                        }
                        lastOutcomeEmail = 'ok';
                        markValid($('#email'), '#emailHelp', '');
                        resolve(true);
                    })
                    .fail(() => {
                        lastOutcomeEmail = '';
                        markInvalid($('#email'), '#emailHelp', 'Não foi possível validar o e-mail.');
                        resolve(false);
                    });
            });
            $('#email').on('blur', validarEmail);

            /* ===== Validação final antes do submit - revalida tudo (usa promises) ===== */
            async function validateAllBeforeSubmit() {
                let ok = true;

                // Nome
                if (isEmpty($('#nome'))) {
                    markInvalid($('#nome'), '#nomeHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#nome'), '#nomeHelp');

                // Nascimento
                if (isEmpty($('#nascimento'))) {
                    markInvalid($('#nascimento'), '#nascimentoHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#nascimento'), '#nascimentoHelp');

                // RG
                if (isEmpty($('#rg'))) {
                    markInvalid($('#rg'), '#rgHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#rg'), '#rgHelp');

                // CPF local + ajax
                const cpfDigits = onlyDigits($('#cpf').val() || '');
                if (cpfDigits.length < 11 || !isValidCPF(cpfDigits)) {
                    markInvalid($('#cpf'), '#cpfHelp', 'CPF inválido.');
                    ok = false;
                } else {
                    clearInvalid($('#cpf'), '#cpfHelp');
                    const cpfOk = await validarCpf();
                    ok = ok && cpfOk;
                }

                // Celular local + ajax
                const celRaw = $('#celular').val() || '';
                if (!isValidCell(celRaw)) {
                    markInvalid($('#celular'), '#celularHelp', 'Informe DDD + número (10 ou 11 dígitos).');
                    ok = false;
                } else {
                    clearInvalid($('#celular'), '#celularHelp');
                    const phoneOk = await validarCelular();
                    ok = ok && phoneOk;
                }

                // Sexo
                if (!$('input[name="sexo"]:checked').length) {
                    markInvalid($('.toggle-group-like'), '#sexoHelp', 'Escolha uma opção');
                    ok = false;
                } else {
                    clearInvalid($('.toggle-group-like'), '#sexoHelp');
                }

                // CEP e endereço
                if (isEmpty($('#cep')) || onlyDigits($('#cep').val()).length !== 8) {
                    markInvalid($('#cep'), '#cepHelp', 'CEP inválido (8 dígitos).');
                    ok = false;
                } else clearInvalid($('#cep'), '#cepHelp');

                if (isEmpty($('#endereco'))) {
                    markInvalid($('#endereco'), '#enderecoHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#endereco'), '#enderecoHelp');
                if (isEmpty($('#numero'))) {
                    markInvalid($('#numero'), '#numeroHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#numero'), '#numeroHelp');
                if (isEmpty($('#bairro'))) {
                    markInvalid($('#bairro'), '#bairroHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#bairro'), '#bairroHelp');
                if (isEmpty($('#cidade'))) {
                    markInvalid($('#cidade'), '#cidadeHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#cidade'), '#cidadeHelp');
                if (isEmpty($('#estado')) || $('#estado').val().length !== 2) {
                    markInvalid($('#estado'), '#estadoHelp', 'Informe UF com 2 letras');
                    ok = false;
                } else clearInvalid($('#estado'), '#estadoHelp');

                // Email local + ajax
                const emailVal = ($('#email').val() || '').trim().toLowerCase();
                if (!isValidEmail(emailVal)) {
                    markInvalid($('#email'), '#emailHelp', 'E-mail inválido.');
                    ok = false;
                } else {
                    clearInvalid($('#email'), '#emailHelp');
                    const emailOk = await validarEmail();
                    ok = ok && emailOk;
                }

                // Senhas
                const p = $('#password').val() || '',
                    pc = $('#password_confirmation').val() || '';
                if (p.length < 8) {
                    markInvalid($('#password'), '#passHelp', 'Mínimo 8 caracteres.');
                    ok = false;
                } else clearInvalid($('#password'), '#passHelp');
                if (pc.length < 8 || p !== pc) {
                    markInvalid($('#password_confirmation'), '#passwordConfirmationHelp', 'Senhas não conferem.');
                    ok = false;
                } else clearInvalid($('#password_confirmation'), '#passwordConfirmationHelp');

                return ok;
            }

            /* ===== Submit handler ===== */
            $('#registerForm').on('submit', async function(e) {
                e.preventDefault();
                $('#btnSubmit').attr('disabled', true);
                const ok = await validateAllBeforeSubmit();
                if (!ok) {
                    $('#btnSubmit').attr('disabled', false);
                    // rolar para primeiro erro visível
                    const $firstErr = $('.is-invalid').first();
                    if ($firstErr && $firstErr.length) {
                        $('html,body').animate({
                            scrollTop: $firstErr.offset().top - 80
                        }, 300);
                    }
                    return;
                }
                // Se tudo ok, submete o formulário (sem AJAX)
                this.submit();
            });

            /* Reset: limpa estados visuais */
            $('#btnReset').on('click', function() {
                $('input').removeClass('is-invalid is-valid');
                $('small.text-danger').text('');
            });

            // pequenas melhorias: revalida quando trocam os campos
            $('#email').on('input', function() {
                clearInvalid($('#email'), '#emailHelp');
                lastEmailChecked = '';
                lastOutcomeEmail = '';
            });
            $('#cpf').on('input', function() {
                clearInvalid($('#cpf'), '#cpfHelp');
                lastCpfChecked = '';
                lastOutcomeCpf = '';
            });
            $('#celular').on('input', function() {
                clearInvalid($('#celular'), '#celularHelp');
                lastPhoneChecked = '';
                lastOutcomePhone = '';
            });

        })();
    </script>
@endsection
