<!doctype html>
<html lang="pt-BR">
<meta name="csrf-token" content="{{ csrf_token() }}">

<head>
    <meta charset="utf-8" />
    <title>Etiqueta 10x4 cm com QR Code</title>
    <style>
        /* 🔧 Tamanho físico exato da etiqueta */
        @page {
            size: 100mm 40mm;
            margin: 0;
        }

        html,
        body {
            margin: 0;
            padding: 0;
            width: 100mm;
            height: 40mm;
            background: #fff;
            overflow: hidden;
            font-family: "Segoe UI", Arial, sans-serif;
            -webkit-print-color-adjust: exact;
            print-color-adjust: exact;
        }

        /* Container */
        .label {
            width: 100mm;
            height: 40mm;
            display: flex;
            align-items: center;
            justify-content: flex-start;
            padding: 0 4mm;
            box-sizing: border-box;
            gap: 5mm;
        }

        /* QR Code */
        .qr {
            width: 28mm;
            height: 28mm;
            display: grid;
            place-items: center;
            flex: 0 0 auto;
        }

        .qr canvas,
        .qr img {
            width: 100%;
            height: 100%;
        }

        /* Texto */
        .info {
            display: flex;
            flex-direction: column;
            justify-content: center;
            line-height: 1.1;
            flex: 1;
            overflow: hidden;
        }

        .nome {
            font-size: 14pt;
            font-weight: 700;
            margin: 0;
            line-height: 1.1em;
            overflow: hidden;
            text-overflow: ellipsis;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            white-space: normal;
        }

        .cargo {
            font-size: 11pt;
            margin: 2px 0 0;
        }

        .empresa {
            font-size: 10pt;
            opacity: 0.8;
            margin: 2px 0 0;
        }

        @media print {
            @page {
                size: 100mm 40mm;
                margin: 0;
            }

            body {
                zoom: 1;
                /* garante 100% */
            }


            .label {
                border: none;
                padding: 0 3mm;
            }
        }
    </style>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/qrcodejs/1.0.0/qrcode.min.js" defer></script>
</head>

<body>
    <div class="label">
        <div class="qr" id="qrcode-1"></div>
        <div class="info">
            <h1 class="nome">Anderson Minervino da Silva</h1>
            <p class="cargo">Evento Nacional</p>
            <p class="empresa">Acesse: linktr.ee/fenasdetran</p>
        </div>
    </div>

    <script>
        // ============== CONFIG ==============
        const API_PENDENTES = '/etiquetas/pendentes'; // GET -> {id, nome, campos}
        const POLL_MS = 5000;
        const PRINT_TIMEOUT = 2000; // fallback se 'afterprint' não disparar

        // ============== ESTADO ==============
        let imprimindo = false;
        const fila = [];
        const idsEnfileirados = new Set();

        // ============== UTILS ==============
        function limparQR(el) {
            el.innerHTML = '';
        }

        function gerarQR(el, texto) {
            return new QRCode(el, {
                text: String(texto || ''),
                width: 280,
                height: 280,
                correctLevel: QRCode.CorrectLevel.H
            });
        }

        function safeParseJSON(str, fallback = {}) {
            if (!str) return fallback;
            try {
                return JSON.parse(str);
            } catch {
                return fallback;
            }
        }

        function preencherEtiqueta({
            nome,
            cargo,
            empresa
        }) {
            document.querySelector('.nome').textContent = nome || '';
            document.querySelector('.cargo').textContent = cargo || '';
            document.querySelector('.empresa').textContent = empresa || '';
        }

        async function fetchJSON(url, options = {}) {
            const opts = {
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    ...(options.headers || {})
                },
                credentials: 'same-origin',
                ...options
            };
            const res = await fetch(url, opts);
            if (res.status === 204) return null;

            const ct = res.headers.get('Content-Type') || '';
            if (!res.ok) throw new Error(`HTTP ${res.status}`);
            if (ct.includes('application/json')) return res.json();
            const txt = await res.text();
            try {
                return JSON.parse(txt);
            } catch {
                return null;
            }
        }

        // Aguarda o DOM realmente pintar (evita “imprimir o anterior”)
        function waitRendered(ms = 120) {
            return new Promise(resolve => {
                requestAnimationFrame(() => {
                    requestAnimationFrame(() => {
                        setTimeout(resolve, ms); // pequeno respiro p/ canvas/img
                    });
                });
            });
        }

        // ============== IMPRESSÃO ==============
        async function processarFila() {
            if (imprimindo || fila.length === 0) return;
            imprimindo = true;

            const item = fila.shift(); // {id, nome, campos}
            if (item) idsEnfileirados.delete(item.id);

            const qrDiv = document.getElementById('qrcode-1');

            // Extrai campos personalizados
            const campos = safeParseJSON(item?.campos, {});
            const cargo = campos.cargo_funcao || campos.cargo || '';
            const empresa = campos.instituicao_trabalho || campos.empresa || campos.recibo || '';

            // QR: prioriza 'qr' ou 'link'; fallback FENASDETRAN
            const qrText = campos.qr || campos.link || 'https://linktr.ee/fenasdetran';

            // Atualiza DOM
            preencherEtiqueta({
                nome: item?.nome || '',
                cargo,
                empresa
            });
            limparQR(qrDiv);
            gerarQR(qrDiv, qrText);

            // *** ponto-chave: aguarde renderização ***
            await waitRendered(150);

            // Callbacks
            let fallbackTimer = null;
            const finalizar = () => {
                if (!imprimindo) return; // já finalizado
                imprimindo = false;
                if (fallbackTimer) clearTimeout(fallbackTimer);
                processarFila();
            };

            const afterprintHandler = () => {
                window.removeEventListener('afterprint', afterprintHandler);
                setTimeout(finalizar, 60);
            };
            window.addEventListener('afterprint', afterprintHandler);

            // Fallback (Firefox ou drivers que não disparam afterprint)
            fallbackTimer = setTimeout(() => {
                window.removeEventListener('afterprint', afterprintHandler);
                finalizar();
            }, PRINT_TIMEOUT);

            // Dispara a impressão (com pequeno atraso para garantir buffer do driver)
            setTimeout(() => window.print(), 80);
        }

        // ============== POLLING ==============
        async function checarPendentes() {
            if (imprimindo) return;
            try {
                const item = await fetchJSON(API_PENDENTES, {
                    method: 'GET'
                });
                if (item && item.id && !idsEnfileirados.has(item.id)) {
                    idsEnfileirados.add(item.id);
                    fila.push(item);
                    processarFila();
                }
            } catch (e) {
                console.warn('Erro ao verificar etiquetas:', e);
            }
        }

        // ============== START ==============
        window.addEventListener('load', () => {
            checarPendentes(); // primeira checagem
            setInterval(checarPendentes, POLL_MS); // loop
        });
    </script>


</body>

</html>
