@extends('site.base')
@section('titulo', 'Atualizar Cadastro - ' . env('EMPRESA'))

@section('content')
    <div class="relative w-full">
        {{-- FORM --}}
        <div class="pt-[64px]">
            <div class="relative w-[95%] md:w-4/6 lg:w-3/6 xl:w-[40%] 4xl:w-[25%] my-6 mx-auto">
                <div class="overflow-auto rounded-2xl shadow-lg flex flex-col w-full h-full bg-white">
                    <div class="h-full p-6">
                        <h1 class="text-primary-green font-bold text-2xl mb-3 mt-6">Atualizar Cadastro</h1>
                        <p class="text-gray-500 mb-6 text-sm">Mantenha seus dados sempre atualizados.</p>

                        <form id="updateForm" action="{{ route('dados.update') }}" method="POST" novalidate enctype="multipart/form-data">
                            @csrf
                            {{-- Foto --}}
                            <div class="mb-6 text-center">
                                <label class="mb-2 text-primary-green font-semibold block">Foto de Perfil</label>
                                {{-- Imagem atual ou padrão --}}
                                <img src="{{ Auth::user()->foto ? Auth::user()->foto : url('images/sem-foto.jpeg') }}" id="selectedImage" width="150" style="border-radius: 20px; cursor:pointer; object-fit: cover; border: 3px solid #16a34a;"
                                    onclick="document.getElementById('foto').click();" alt="Foto de perfil">
                                {{-- Input oculto --}}
                                <input type="file" id="foto" name="foto" accept="image/*" class="hidden" onchange="previewImage(event)">
                                @error('foto')
                                    <small class="text-red-500">{{ $message }}</small>
                                @enderror
                            </div>

                            {{-- Script de preview --}}
                            <script>
                                function previewImage(event) {
                                    const file = event.target.files[0];
                                    if (file) {
                                        const reader = new FileReader();
                                        reader.onload = function(e) {
                                            document.getElementById('selectedImage').src = e.target.result;
                                        };
                                        reader.readAsDataURL(file);
                                    }
                                }
                            </script>

                            {{-- Nome --}}
                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">Nome completo</label>
                                <input type="text" id="nome" name="nome" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('nome') ? 'is-invalid' : '' }}"
                                    value="{{ old('nome', Auth::user()->name . ' ' . Auth::user()->sobrenome) }}">
                                @error('nome')
                                    <small class="text-red-500">{{ $message }}</small>
                                @enderror
                            </div>

                            {{-- RG / CPF --}}
                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">RG</label>
                                    <input type="text" id="rg" name="rg" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('rg') ? 'is-invalid' : '' }}"
                                        value="{{ old('rg', Auth::user()->rg) }}">
                                    @error('rg')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">CPF</label>
                                    <input type="text" id="cpf" name="cpf" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('cpf') ? 'is-invalid' : '' }}" value="{{ Auth::user()->cpf }}">
                                    @error('cpf')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                            </div>

                            {{-- Celular / Nascimento --}}
                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Celular (WhatsApp)</label>
                                    <input type="tel" id="celular" name="celular" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('celular') ? 'is-invalid' : '' }}"
                                        value="{{ old('celular', Auth::user()->celular) }}">
                                    @error('celular')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Data de Nascimento</label>
                                    <input type="date" id="nascimento" name="nascimento" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('nascimento') ? 'is-invalid' : '' }}"
                                        value="{{ old('nascimento', Auth::user()->nascimento) }}">
                                    @error('nascimento')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                            </div>

                            {{-- Sexo --}}
                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">Sexo</label>
                                <div class="flex gap-4 mt-2">
                                    <label><input type="radio" name="sexo" value="Masculino" {{ old('sexo', Auth::user()->sexo) == 'Masculino' ? 'checked' : '' }}> Masculino</label>
                                    <label><input type="radio" name="sexo" value="Feminino" {{ old('sexo', Auth::user()->sexo) == 'Feminino' ? 'checked' : '' }}> Feminino</label>
                                </div>
                                @error('sexo')
                                    <small class="text-red-500">{{ $message }}</small>
                                @enderror
                            </div>

                            {{-- Endereço --}}
                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">CEP</label>
                                <input type="text" id="cep" name="cep" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('cep') ? 'is-invalid' : '' }}"
                                    value="{{ old('cep', Auth::user()->cep) }}">
                                @error('cep')
                                    <small class="text-red-500">{{ $message }}</small>
                                @enderror
                            </div>

                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">Endereço</label>
                                <input type="text" id="endereco" name="endereco" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('endereco') ? 'is-invalid' : '' }}"
                                    value="{{ old('endereco', Auth::user()->endereco) }}">
                                @error('endereco')
                                    <small class="text-red-500">{{ $message }}</small>
                                @enderror
                            </div>

                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Número</label>
                                    <input type="text" id="numero" name="numero" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('numero') ? 'is-invalid' : '' }}"
                                        value="{{ old('numero', Auth::user()->numero) }}">
                                    @error('numero')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Bairro</label>
                                    <input type="text" id="bairro" name="bairro" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('bairro') ? 'is-invalid' : '' }}"
                                        value="{{ old('bairro', Auth::user()->bairro) }}">
                                    @error('bairro')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                            </div>

                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Cidade</label>
                                    <input type="text" id="cidade" name="cidade" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('cidade') ? 'is-invalid' : '' }}"
                                        value="{{ old('cidade', Auth::user()->cidade) }}">
                                    @error('cidade')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Estado (UF)</label>
                                    <input type="text" id="estado" name="estado" maxlength="2" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('estado') ? 'is-invalid' : '' }}"
                                        value="{{ old('estado', Auth::user()->estado) }}">
                                    @error('estado')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                            </div>

                            {{-- Email --}}
                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">E-mail</label>
                                <input type="email" id="email" name="email" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('email') ? 'is-invalid' : '' }}"
                                    value="{{ old('email', Auth::user()->email) }}">
                                @error('email')
                                    <small class="text-red-500">{{ $message }}</small>
                                @enderror
                            </div>

                            {{-- Senhas --}}
                            <div class="flex flex-col md:flex-row gap-3 mb-4">
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Senha</label>
                                    <input type="password" id="password" name="password" class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('password') ? 'is-invalid' : '' }}">
                                    @error('password')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                                <div class="w-full">
                                    <label class="mb-1 text-primary-green">Confirmar Senha</label>
                                    <input type="password" id="password_confirmation" name="password_confirmation"
                                        class="bg-green-light w-full h-11 px-4 rounded-full text-base focus:outline-none {{ $errors->has('password_confirmation') ? 'is-invalid' : '' }}">
                                    @error('password_confirmation')
                                        <small class="text-red-500">{{ $message }}</small>
                                    @enderror
                                </div>
                            </div>

                            {{-- Botão --}}
                            <div class="flex justify-center mt-6">
                                <button type="submit" id="btnSubmit"
                                    class="bg-primary-green w-full flex hover:bg-primary-green/90
               rounded-full px-4 py-[0.6rem] disabled:bg-disabled
               disabled:cursor-not-allowed relative justify-center items-center">

                                    <!-- Texto -->
                                    <span id="btnSubmitText" class="text-center text-white">Salvar alterações</span>

                                    <!-- Loader 3 dots -->
                                    <span id="btnSubmitLoader" class="hidden flex gap-3 items-center justify-center">
                                        <span class="dot"></span>
                                        <span class="dot delay-150"></span>
                                        <span class="dot delay-300"></span>
                                    </span>
                                </button>
                            </div>


                        </form>

                        {{-- feedback geral --}}
                        @if ($errors->any())
                            <div class="alert alert-danger mt-3">
                                <strong>Ops!</strong> Verifique os campos destacados e tente novamente.
                            </div>
                        @endif
                        @if (session('success'))
                            <div class="alert alert-success mt-3">{{ session('success') }}</div>
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection
@section('js')
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>

    <script>
        document.addEventListener("DOMContentLoaded", function() {
            const form = document.getElementById("updateForm");
            const btn = document.getElementById("btnSubmit");
            const text = document.getElementById("btnSubmitText");
            const loader = document.getElementById("btnSubmitLoader");

            if (form && btn) {
                form.addEventListener("submit", function() {
                    btn.disabled = true;
                    btn.classList.remove("bg-primary-green", "hover:bg-primary-green/90");
                    btn.style.background = "none";

                    text.classList.add("hidden");
                    loader.classList.remove("hidden");
                });
            }
        });

        (function() {
            const CSRF = $('meta[name="csrf-token"]').attr('content');

            /* ===== Helpers ===== */
            function onlyDigits(v) {
                return (v || '').replace(/\D/g, '');
            }

            function markInvalid($el, helpSel, msg) {
                $el.addClass('is-invalid').removeClass('is-valid');
                if (helpSel) $(helpSel).text(msg || 'Campo obrigatório');
            }

            function markValid($el, helpSel, msg) {
                $el.addClass('is-valid').removeClass('is-invalid');
                if (helpSel) $(helpSel).text(msg || '');
            }

            function clearInvalid($el, helpSel) {
                $el.removeClass('is-invalid is-valid');
                if (helpSel) $(helpSel).text('');
            }

            function isEmpty($el) {
                return !($el.val() || '').toString().trim();
            }

            /* ===== Máscaras ===== */
            $('#cpf').on('input', function() {
                let v = onlyDigits($(this).val()).slice(0, 11);
                v = v.replace(/(\d{3})(\d)/, '$1.$2');
                v = v.replace(/(\d{3})(\d)/, '$1.$2');
                v = v.replace(/(\d{3})(\d{1,2})$/, '$1-$2');
                $(this).val(v);
            });

            $('#celular').on('input', function() {
                let v = onlyDigits($(this).val()).slice(0, 11);
                if (v.length >= 11) v = v.replace(/(\d{2})(\d{5})(\d{0,4})/, '($1) $2-$3');
                else if (v.length >= 10) v = v.replace(/(\d{2})(\d{4})(\d{0,4})/, '($1) $2-$3');
                else if (v.length > 2) v = v.replace(/(\d{2})(\d{0,5})/, '($1) $2');
                else if (v.length > 0) v = v.replace(/(\d{0,2})/, '($1');
                $(this).val(v);
                clearInvalid($('#celular'), '#celularHelp');
            });

            $('#cep').on('input', function() {
                let v = onlyDigits($(this).val()).slice(0, 8);
                if (v.length > 5) v = v.slice(0, 5) + '-' + v.slice(5);
                $(this).val(v);
            });

            $('#estado').on('input', function() {
                this.value = this.value.toUpperCase().slice(0, 2);
            });

            /* ===== Validações locais ===== */
            function isValidCPF(c) {
                c = onlyDigits(c);
                if (!c || c.length !== 11 || /^(\d)\1+$/.test(c)) return false;
                let s = 0,
                    r;
                for (let i = 1; i <= 9; i++) s += parseInt(c.substring(i - 1, i)) * (11 - i);
                r = (s * 10) % 11;
                if (r == 10 || r == 11) r = 0;
                if (r != parseInt(c.substring(9, 10))) return false;
                s = 0;
                for (let i = 1; i <= 10; i++) s += parseInt(c.substring(i - 1, i)) * (12 - i);
                r = (s * 10) % 11;
                if (r == 10 || r == 11) r = 0;
                return r == parseInt(c.substring(10, 11));
            }

            function isValidEmail(e) {
                return /^[^\s@]+@[^\s@]+\.[^\s@]{2,}$/.test((e || '').toLowerCase());
            }

            function isValidCell(full) {
                const d = onlyDigits(full);
                return d.length === 10 || d.length === 11;
            }

            function extractDDDAndNumber(full) {
                const d = onlyDigits(full);
                return d.length < 10 ? {
                    ddd: '',
                    numero: ''
                } : {
                    ddd: d.slice(0, 2),
                    numero: d.slice(2)
                };
            }

            /* ===== ViaCEP (jsonp) ===== */
            $('#cep').on('blur', function() {
                const numCep = onlyDigits($('#cep').val());
                if (numCep.length !== 8) {
                    markInvalid($('#cep'), '#cepHelp', 'CEP inválido (8 dígitos).');
                    return;
                }
                $('#cepHelp').text('Consultando CEP...');
                $.ajax({
                    url: 'https://viacep.com.br/ws/' + numCep + '/json/?callback=?',
                    dataType: 'jsonp',
                    success: function(d) {
                        if (d.erro) {
                            markInvalid($('#cep'), '#cepHelp', 'CEP não encontrado.');
                            return;
                        }
                        $('#cep').val(d.cep);
                        $('#endereco').val(d.logradouro || '');
                        $('#bairro').val(d.bairro || '');
                        $('#cidade').val(d.localidade || '');
                        $('#estado').val(d.uf || '');
                        markValid($('#cep'), '#cepHelp', '');
                    },
                    error: function() {
                        markInvalid($('#cep'), '#cepHelp', 'Erro ao consultar CEP.');
                    }
                });
            });

            /* ===== AJAX setup ===== */
            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': CSRF
                }
            });

            /* ===== Async validators (CPF / Celular / Email) - same logic as seu exemplo ===== */
            let lastCpfChecked = '',
                lastOutcomeCpf = '',
                cpfAjax = null;
            let lastPhoneChecked = '',
                lastOutcomePhone = '',
                phoneAjax = null;
            let lastEmailChecked = '',
                lastOutcomeEmail = '',
                emailAjax = null;

            const validarCpf = () => new Promise((resolve) => {
                const raw = $('#cpf').val() || '',
                    digits = onlyDigits(raw);
                if (digits.length < 11 || !isValidCPF(digits)) {
                    markInvalid($('#cpf'), '#cpfHelp', 'CPF inválido.');
                    return resolve(false);
                }
                if (digits === lastCpfChecked && lastOutcomeCpf) {
                    const ok = (lastOutcomeCpf === 'ok');
                    ok ? markValid($('#cpf'), '#cpfHelp') : markInvalid($('#cpf'), '#cpfHelp', 'CPF já cadastrado.');
                    return resolve(ok);
                }
                if (cpfAjax && cpfAjax.readyState !== 4) cpfAjax.abort();
                cpfAjax = $.post("{{ route('validar.cpf') }}", {
                        cpf: raw
                    })
                    .done(res => {
                        lastCpfChecked = digits;
                        if (res === 'cpf_error') {
                            lastOutcomeCpf = 'cpf_error';
                            markInvalid($('#cpf'), '#cpfHelp', 'CPF inválido.');
                            return resolve(false);
                        }
                        if (res === 'cpf_exist') {
                            lastOutcomeCpf = 'cpf_exist';
                            markInvalid($('#cpf'), '#cpfHelp', 'Este CPF já está cadastrado.');
                            return resolve(false);
                        }
                        lastOutcomeCpf = 'ok';
                        markValid($('#cpf'), '#cpfHelp', '');
                        resolve(true);
                    })
                    .fail(() => {
                        lastOutcomeCpf = '';
                        markInvalid($('#cpf'), '#cpfHelp', 'Não foi possível validar o CPF.');
                        resolve(false);
                    });
            });
            $('#cpf').on('blur', validarCpf);

            const validarCelular = () => new Promise((resolve) => {
                const full = $('#celular').val() || '';
                if (!isValidCell(full)) {
                    markInvalid($('#celular'), '#celularHelp', 'Informe DDD + número.');
                    return resolve(false);
                }
                const fullDigits = onlyDigits(full);
                if (fullDigits === lastPhoneChecked && lastOutcomePhone) {
                    const ok = (lastOutcomePhone === 'ok');
                    ok ? markValid($('#celular'), '#celularHelp') : markInvalid($('#celular'), '#celularHelp', 'Celular já cadastrado.');
                    return resolve(ok);
                }
                const {
                    ddd,
                    numero
                } = extractDDDAndNumber(full);
                if (!ddd || !numero) {
                    markInvalid($('#celular'), '#celularHelp', 'Informe DDD + número.');
                    return resolve(false);
                }
                if (phoneAjax && phoneAjax.readyState !== 4) phoneAjax.abort();
                phoneAjax = $.post("{{ route('validar.celular') }}", {
                        ddd,
                        celular: numero
                    })
                    .done(res => {
                        lastPhoneChecked = fullDigits;
                        if (res === 'celular_error') {
                            lastOutcomePhone = 'celular_error';
                            markInvalid($('#celular'), '#celularHelp', 'Número inválido.');
                            return resolve(false);
                        }
                        if (res === 'celular_exist') {
                            lastOutcomePhone = 'celular_exist';
                            markInvalid($('#celular'), '#celularHelp', 'Este celular já está cadastrado.');
                            return resolve(false);
                        }
                        lastOutcomePhone = 'ok';
                        markValid($('#celular'), '#celularHelp', '');
                        resolve(true);
                    })
                    .fail(() => {
                        lastOutcomePhone = '';
                        markInvalid($('#celular'), '#celularHelp', 'Não foi possível validar o celular.');
                        resolve(false);
                    });
            });
            $('#celular').on('blur', validarCelular);

            const validarEmail = () => new Promise((resolve) => {
                const email = ($('#email').val() || '').trim().toLowerCase();
                if (!isValidEmail(email)) {
                    markInvalid($('#email'), '#emailHelp', 'E-mail inválido.');
                    return resolve(false);
                }
                if (email === lastEmailChecked && lastOutcomeEmail) {
                    const ok = (lastOutcomeEmail === 'ok');
                    ok ? markValid($('#email'), '#emailHelp') : markInvalid($('#email'), '#emailHelp', 'E-mail já cadastrado.');
                    return resolve(ok);
                }
                if (emailAjax && emailAjax.readyState !== 4) emailAjax.abort();
                emailAjax = $.post("{{ route('validar.email') }}", {
                        email
                    })
                    .done(res => {
                        lastEmailChecked = email;
                        if (res === 'email_exist') {
                            lastOutcomeEmail = 'email_exist';
                            markInvalid($('#email'), '#emailHelp', 'Este e-mail já está cadastrado.');
                            return resolve(false);
                        }
                        lastOutcomeEmail = 'ok';
                        markValid($('#email'), '#emailHelp', '');
                        resolve(true);
                    })
                    .fail(() => {
                        lastOutcomeEmail = '';
                        markInvalid($('#email'), '#emailHelp', 'Não foi possível validar o e-mail.');
                        resolve(false);
                    });
            });
            $('#email').on('blur', validarEmail);

            /* ===== Validação final antes do submit - revalida tudo (usa promises) ===== */
            async function validateAllBeforeSubmit() {
                let ok = true;

                // Nome
                if (isEmpty($('#nome'))) {
                    markInvalid($('#nome'), '#nomeHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#nome'), '#nomeHelp');

                // Nascimento
                if (isEmpty($('#nascimento'))) {
                    markInvalid($('#nascimento'), '#nascimentoHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#nascimento'), '#nascimentoHelp');

                // RG
                if (isEmpty($('#rg'))) {
                    markInvalid($('#rg'), '#rgHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#rg'), '#rgHelp');

                // CPF local + ajax
                const cpfDigits = onlyDigits($('#cpf').val() || '');
                if (cpfDigits.length < 11 || !isValidCPF(cpfDigits)) {
                    markInvalid($('#cpf'), '#cpfHelp', 'CPF inválido.');
                    ok = false;
                } else {
                    clearInvalid($('#cpf'), '#cpfHelp');
                    const cpfOk = await validarCpf();
                    ok = ok && cpfOk;
                }

                // Celular local + ajax
                const celRaw = $('#celular').val() || '';
                if (!isValidCell(celRaw)) {
                    markInvalid($('#celular'), '#celularHelp', 'Informe DDD + número (10 ou 11 dígitos).');
                    ok = false;
                } else {
                    clearInvalid($('#celular'), '#celularHelp');
                    const phoneOk = await validarCelular();
                    ok = ok && phoneOk;
                }

                // Sexo
                if (!$('input[name="sexo"]:checked').length) {
                    markInvalid($('.toggle-group-like'), '#sexoHelp', 'Escolha uma opção');
                    ok = false;
                } else {
                    clearInvalid($('.toggle-group-like'), '#sexoHelp');
                }

                // CEP e endereço
                if (isEmpty($('#cep')) || onlyDigits($('#cep').val()).length !== 8) {
                    markInvalid($('#cep'), '#cepHelp', 'CEP inválido (8 dígitos).');
                    ok = false;
                } else clearInvalid($('#cep'), '#cepHelp');

                if (isEmpty($('#endereco'))) {
                    markInvalid($('#endereco'), '#enderecoHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#endereco'), '#enderecoHelp');
                if (isEmpty($('#numero'))) {
                    markInvalid($('#numero'), '#numeroHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#numero'), '#numeroHelp');
                if (isEmpty($('#bairro'))) {
                    markInvalid($('#bairro'), '#bairroHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#bairro'), '#bairroHelp');
                if (isEmpty($('#cidade'))) {
                    markInvalid($('#cidade'), '#cidadeHelp', 'Obrigatório');
                    ok = false;
                } else clearInvalid($('#cidade'), '#cidadeHelp');
                if (isEmpty($('#estado')) || $('#estado').val().length !== 2) {
                    markInvalid($('#estado'), '#estadoHelp', 'Informe UF com 2 letras');
                    ok = false;
                } else clearInvalid($('#estado'), '#estadoHelp');

                // Email local + ajax
                const emailVal = ($('#email').val() || '').trim().toLowerCase();
                if (!isValidEmail(emailVal)) {
                    markInvalid($('#email'), '#emailHelp', 'E-mail inválido.');
                    ok = false;
                } else {
                    clearInvalid($('#email'), '#emailHelp');
                    const emailOk = await validarEmail();
                    ok = ok && emailOk;
                }

                // Senhas
                // Senhas (só valida se um dos campos foi preenchido)
                const p = $('#password').val() || '',
                    pc = $('#password_confirmation').val() || '';

                if (p || pc) {
                    if (p.length < 8) {
                        markInvalid($('#password'), '#passHelp', 'Mínimo 8 caracteres.');
                        ok = false;
                    } else clearInvalid($('#password'), '#passHelp');

                    if (pc.length < 8 || p !== pc) {
                        markInvalid($('#password_confirmation'), '#passwordConfirmationHelp', 'Senhas não conferem.');
                        ok = false;
                    } else clearInvalid($('#password_confirmation'), '#passwordConfirmationHelp');
                } else {
                    clearInvalid($('#password'), '#passHelp');
                    clearInvalid($('#password_confirmation'), '#passwordConfirmationHelp');
                }


                return ok;
            }

            /* ===== Submit handler ===== */
            $('#registerForm').on('submit', async function(e) {
                e.preventDefault();
                $('#btnSubmit').attr('disabled', true);
                const ok = await validateAllBeforeSubmit();
                if (!ok) {
                    $('#btnSubmit').attr('disabled', false);
                    // rolar para primeiro erro visível
                    const $firstErr = $('.is-invalid').first();
                    if ($firstErr && $firstErr.length) {
                        $('html,body').animate({
                            scrollTop: $firstErr.offset().top - 80
                        }, 300);
                    }
                    return;
                }
                // Se tudo ok, submete o formulário (sem AJAX)
                this.submit();
            });

            /* Reset: limpa estados visuais */
            $('#btnReset').on('click', function() {
                $('input').removeClass('is-invalid is-valid');
                $('small.text-red-500').text('');
            });

            // pequenas melhorias: revalida quando trocam os campos
            $('#email').on('input', function() {
                clearInvalid($('#email'), '#emailHelp');
                lastEmailChecked = '';
                lastOutcomeEmail = '';
            });
            $('#cpf').on('input', function() {
                clearInvalid($('#cpf'), '#cpfHelp');
                lastCpfChecked = '';
                lastOutcomeCpf = '';
            });
            $('#celular').on('input', function() {
                clearInvalid($('#celular'), '#celularHelp');
                lastPhoneChecked = '';
                lastOutcomePhone = '';
            });

        })();

        function btnSubmit() {
            var timerInterval;
            Swal.fire({
                title: "Atualizando seus dados",
                html: "Pro favor aguarde... <b></b>",
                timer: 10000,
                timerProgressBar: true,
                didOpen: function() {
                    Swal.showLoading();
                    var b = Swal.getPopup().querySelector("b");
                    timerInterval = setInterval(function() {
                        var left = Swal.getTimerLeft ? Swal.getTimerLeft() : 0;
                        if (b) {
                            b.textContent = Math.ceil(left / 1000) + "s";
                        }
                    }, 100);
                },
                willClose: function() {
                    clearInterval(timerInterval);
                }
            });
        }

        document.getElementById('updateForm').addEventListener('submit', function(e) {
            btnSubmit(); // chama seu SweetAlert
        });

        @if (session('success'))
            Swal.fire({
                icon: 'success',
                title: 'Sucesso!',
                text: '{{ session('success') }}',
                confirmButtonText: 'OK',
                confirmButtonColor: '#16a34a' // Verde Tailwind (bg-green-600)
            });
            setInterval(() => {
                Swal.close();
            }, 2000);
        @endif

        @if (session('error'))
            Swal.fire({
                icon: 'error',
                title: 'Erro!',
                text: '{{ session('error') }}',
                confirmButtonText: 'Fechar',
                confirmButtonColor: '#dc2626' // Vermelho Tailwind (bg-red-600)
            });
        @endif
    </script>
@endsection
