<?php

namespace App\Providers;

use App\Models\Analytics;
use App\Models\Estado;
use App\Models\Categoria;
use Carbon\Carbon;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Request;

class AppServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        //
    }

    public function boot(): void
    {
        // Variáveis globais disponíveis em todas as views
        view()->composer('*', function ($view) {

            //CONTROLANDO O Analytics
            $request    = Request::instance();
            $eventoId   = $request->route('id');

            try {
                $sessionId = $request->session()->getId();
                $userId = auth()->check() ? auth()->id() : null;
                $url = $request->fullUrl(); // ou $request->url() se quiser ignorar query params
                $hoje = Carbon::today();

                $acesso = Analytics::where('url', $url)
                    ->whereDate('created_at', $hoje)
                    ->where(function ($query) use ($sessionId, $userId) {
                        if ($userId) {
                            $query->where('user_id', $userId);
                        } else {
                            $query->where('session_id', $sessionId);
                        }
                    })
                    ->first();

                if ($acesso) {
                    // Já registrou hoje, incrementa
                    $acesso->increment('views');
                } else {
                    // Primeira visita de hoje
                    Analytics::create([
                        'session_id' => $sessionId,
                        'user_id'    => $userId,
                        'evento_id'  => $eventoId,
                        'url'        => $url,
                        'ip_address' => $request->ip(),
                        'user_agent' => $request->userAgent(),
                        'referer'    => $request->headers->get('referer'),
                        'views'      => 1,
                    ]);
                }
            } catch (\Throwable $e) {
                // logger()->error($e->getMessage());
            }


            // Cache eterno para listas fixas
            $estados    = cache()->rememberForever('estados', fn() => Estado::all());
            $categorias = cache()->rememberForever('categorias', fn() => Categoria::all());

            // Saudação automática
            $horaAtual = now()->format('H');
            if ($horaAtual >= 6 && $horaAtual < 12) {
                $saudacao = 'Bom dia';
            } elseif ($horaAtual >= 12 && $horaAtual < 18) {
                $saudacao = 'Boa tarde';
            } else {
                $saudacao = 'Boa noite';
            }

            // Envia para as views
            $view->with([
                'estados'           => $estados,
                'categorias'        => $categorias,
                'saudacao'          => $saudacao,
                'status'            => config('status.geral'),
                'status_pagamento'  => config('status.pagamento'),
                'forma_pagamento'   => config('status.forma_pagamento'),
                'status_convite'    => config('status.convite'),
                'cor_status_convite' => config('status.cor_convite'),
                'nivel_colaborador' => config('status.nivel_colaborador'),
                'status_pdv'        => config('status.pdv'),
                'cor_status_pdv'    => config('status.cor_pdv'),
                'status_ingresso'   => config('status.ingresso'),
                'cor_status_ingresso' => config('status.cor_ingresso'),
                'status_checkin'    => config('status.checkin'),
                'cor_status_checkin' => config('status.cor_checkin'),
                'status_boleto'     => config('status.boleto'),
                'cor_status_boleto' => config('status.cor_boleto'),
                'mes'               => config('status.mes'),
                'semana'            => config('status.semana'),
            ]);
        });

        // Ajuste padrão para compatibilidade MySQL
        Schema::defaultStringLength(191);
    }
}
