<?php

namespace App\Http\Controllers;

use App\Models\Categoria;
use App\Models\Estado;
use App\Models\Evento;
use App\Models\EventoCupomDesconto;
use App\Models\EventoPacote;
use App\Models\GetWay;
use App\Models\Ingresso;
use App\Models\Locais;
use App\Models\Pedido;
use App\Models\PlayList;
use App\Models\TokenPush;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Milon\Barcode\DNS1D;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Redirect;

class SiteController extends Controller
{

    public function solicitar()
    {
        $Evento = Evento::where('status', 'Ativo');

        dd($Evento->getOriginal());
    }

    public function suporte($user_id = null)
    {
        $whatsapp = env('WHATSAPP'); // Número do WhatsApp no .env, ex: 71999998888
        if ($user_id) {
            $User = User::find($user_id);
            $mensagem = "Olá, eu me chamo {$User->name}, estou no APP da UCESUB e preciso de ajuda para minha conta. Meu código do cliente UCESUB é {$User->id}.";
        } else {
            $mensagem = "Olá estou no APP da UCESUB, preciso de ajuda.";
        }

        $url = "https://api.whatsapp.com/send?1=pt_BR&phone=55" . setUri2($whatsapp) . "&text=" . urlencode($mensagem);
        return redirect()->away($url);
    }

    protected function sendPushNotification($expoPushToken, $title, $body, $data = [])
    {
        $pushController = app(PushProdutoController::class);
        $request = new Request([
            'expoPushToken' => $expoPushToken,
            'title' => $title,
            'body' => $body,
            'data' => $data,
        ]);

        $pushController->sendNotification($request);
    }

    private function salvarCarrinhoNoBancoDeDados(Request $request)
    {
        // Salva o carrinho no banco de dados de sessões
        $sessionId = $request->session()->getId();
        $payload = base64_encode(serialize($request->session()->all()));

        DB::table('sessions')->updateOrInsert(
            ['id' => $sessionId],
            ['payload' => $payload, 'last_activity' => time()]
        );
    }


    public function home()
    {

        $visitados = session()->get('visitados');

        $hojeData = Carbon::today()->toDateString();
        $inicio   = Carbon::today();
        $fim      = Carbon::tomorrow()->setHour(3);

        // cache de 1 dia (1440 minutos)
        $top = Cache::remember('eventos_top', 1440, function () {
            return Evento::where('status', 'Ativo')
                ->where('tipo', 'Público')
                ->where('data_fim', '>', now())
                ->orderBy('views', 'desc')
                ->get();
        });

        $eventos = Cache::remember('eventos_listagem', 1440, function () {
            return Evento::where('status', 'Ativo')
                ->where('tipo', 'Público')
                ->where('data_fim', '>', now())
                ->orderBy('data_inicio', 'asc')
                ->get();
        });

        $hoje = Cache::remember('eventos_hoje', 1440, function () use ($hojeData, $fim) {
            return Evento::where('status', 'Ativo')
                ->where('tipo', 'Público')
                ->where(function ($query) use ($hojeData, $fim) {
                    $query->whereDate('data_inicio', $hojeData)
                        ->orWhere('data_fim', '<=', $fim);
                })
                ->orderBy('data_inicio', 'asc')
                ->get();
        });

        return view('home.home', compact('eventos', 'top', 'hoje', 'visitados'));
    }

    public function semana()
    {
        // Define o intervalo da semana atual
        $inicioSemana = now()->startOfWeek(Carbon::MONDAY)->startOfDay(); // Segunda 00:00
        $fimSemana = now()->endOfWeek(Carbon::SUNDAY)->endOfDay(); // Domingo 23:59

        // Armazena em cache os eventos da semana
        $Eventos = Cache::remember('eventos_publicos_semana', 60, function () use ($inicioSemana, $fimSemana) {
            return Evento::where('status', 'Ativo')
                ->where('tipo', 'Público')
                ->whereBetween('data_inicio', [$inicioSemana, $fimSemana])
                ->orWhereBetween('data_fim', [$inicioSemana, $fimSemana])
                ->orderBy('data_inicio', 'asc')
                ->get();
        });

        return view('site.eventos', compact('Eventos'));
    }


    public function eventos()
    {
        $Evento = Evento::where('status', 'Ativo')
            ->where('tipo', 'Público')
            ->where('data_fim', '>', now())
            ->orderBy('id', 'desc')
            ->first();


        return redirect()->route('site.evento', [$Evento->url, $Evento->id]);
    }

    public function evento(Request $request, $url, $id, $cupom = null)
    {
        // CUPOM
        if ($cupom) {
            $exe = EventoCupomDesconto::where('evento_id', $id)
                ->where('cupom', $cupom)
                ->first();

            if ($exe) {
                $ingressosUsados = Ingresso::where('evento_id', $id)
                    ->where('cupom_id', $exe->id)
                    ->count();

                if ($exe->limite_ingressos > 0 && $ingressosUsados >= $exe->limite_ingressos) {
                    session()->forget('cupom');
                } else {
                    $cupomData = [
                        'id' => $exe->id,
                        'evento_id'     => $exe->evento_id,
                        'cupom'         => $exe->cupom,
                        'desconto'      => $exe->desconto,
                        'situacao'      => $exe->situacao,
                    ];
                    $request->session()->put('cupom', $cupomData);
                    $this->salvarCarrinhoNoBancoDeDados($request);
                }
            } else {
                session()->forget('cupom');
            }
        }

        $cupom = session()->get('cupom');

        $evento = Evento::where('id', $id)
            ->where('status', 'Ativo')
            ->where('tipo', 'Público')
            ->first();

        if (empty($evento)) {
            return view('site.404');
        }

        // CUPOM realmente pertence ao evento?
        if ($cupom && $cupom['evento_id'] != $evento->id) {
            session()->forget('cupom');
            $cupom = null;
        }

        $evento->update(['views' => $evento->views + 1]);

        $dataAtual = Carbon::now();

        // =========================
        // VERIFICAÇÃO DOS PACOTES
        // =========================
        $pacotes = $evento->Pacotes()
            ->where('preco', '>', 0)
            ->orderBy('id')
            ->get();

        $temPacoteAtivoComIngresso = false;

        foreach ($pacotes as $pacote) {
            $vendidos = $pacote->Ingressos->where('status', '1')->count();
            $restantes = $pacote->qtd - $vendidos;

            if ($pacote->status === 'Ativo' && $restantes > 0) {
                $temPacoteAtivoComIngresso = true;
                break;
            }
        }

        // Ativar próximo pacote SOMENTE se todos os ativos estão esgotados
        if ($evento->notificacao_push == '0') {
            if (!$temPacoteAtivoComIngresso) {
                $pacoteInativo = $evento->Pacotes()
                    ->where('status', 'Inativo')
                    ->where('preco', '>', 0)
                    ->orderBy('id')
                    ->first();

                if ($pacoteInativo) {
                    $pacoteInativo->update([
                        'status' => 'Ativo',
                        'data_inicio' => $dataAtual->format('Y-m-d H:i:s')
                    ]);

                    $tokens = TokenPush::where('user_id', 1)->get();
                    foreach ($tokens as $push) {
                        if ($push->token) {
                            $this->sendPushNotification(
                                $push->token,
                                $evento->titulo,
                                'Novo lote "' . $pacoteInativo->nome . '" ativado automaticamente.',
                                [
                                    'evento_id' => $evento->id,
                                    'lote_id' => $pacoteInativo->id,
                                    'tipo' => 'novo_lote'
                                ]
                            );
                        }
                    }
                    return redirect()->route('site.evento', [$evento->url, $evento->id]);
                }

                // Se nenhum inativo foi encontrado, significa que não há ingressos
                $tokens = TokenPush::where('user_id', 1)->get();
                foreach ($tokens as $push) {
                    if ($push->token) {
                        $this->sendPushNotification(
                            $push->token,
                            $evento->titulo,
                            'O evento está sem ingressos disponíveis.',
                            [
                                'evento_id' => $evento->id,
                                'tipo' => 'sem_ingressos'
                            ]
                        );
                    }
                }
            }
        }

        // =========================
        // FORMATANDO DATA E SALVANDO VISITADOS NA SESSÃO
        // =========================
        $mes = [
            "01" => 'Janeiro',
            "1" => 'Janeiro',
            "02" => 'Fevereiro',
            "2" => 'Fevereiro',
            "03" => 'Março',
            "3" => 'Março',
            "04" => 'Abril',
            "4" => 'Abril',
            "05" => 'Maio',
            "5" => 'Maio',
            "06" => 'Junho',
            "6" => 'Junho',
            "07" => 'Julho',
            "7" => 'Julho',
            "08" => 'Agosto',
            "8" => 'Agosto',
            "09" => 'Setembro',
            "9" => 'Setembro',
            "10" => 'Outubro',
            "11" => 'Novembro',
            "12" => 'Dezembro',
        ];

        $semana = [
            '0' => 'Domingo',
            '1' => 'Segunda',
            '2' => 'Terça',
            '3' => 'Quarta',
            '4' => 'Quinta',
            '5' => 'Sexta',
            '6' => 'Sábado',
        ];

        $visitados = $request->session()->get('visitados', []);
        $visitados[$evento->id] = [
            'evento_id' => $evento->id,
            'titulo' => $evento->titulo,
            'banner' => $evento->banner,
            'local' => $evento->local . ', ' . $evento->cidade . '-' . $evento->estado,
            'data' => mb_substr($semana[date('w', strtotime($evento->data_inicio))], 0, 3, 'UTF-8')
                . ' ' . date('d', strtotime($evento->data_inicio))
                . ' de ' . substr($mes[date('m', strtotime($evento->data_inicio))], 0, 3),
            'url' => $evento->url,
        ];

        $request->session()->put('visitados', $visitados);
        $this->salvarCarrinhoNoBancoDeDados($request);
        return view('site.evento', ['evento' => $evento, 'cupom' => $cupom]);
    }


    public function divulgador(Request $request, $url, $id, $divulgador = null)
    {
        // DIVULGADOR
        if ($divulgador) {
            $user = User::find($divulgador);

            if ($user) {
                $divulgadorData = [
                    'id' => $user->id,
                    'nome' => $user->name,
                    'email' => $user->email,
                ];
                $request->session()->put('divulgador', $divulgadorData);
                $this->salvarCarrinhoNoBancoDeDados($request);
            } else {
                session()->forget('divulgador');
            }
        }

        $divulgador = session()->get('divulgador');

        $evento = Evento::where('id', $id)
            ->where('status', 'Ativo')
            ->where('tipo', 'Público')
            ->first();

        if (empty($evento)) {
            return view('site.404');
        }

        // Se o divulgador estiver setado e o evento não existir mais
        if ($divulgador && !$evento) {
            session()->forget('divulgador');
            $divulgador = null;
        }

        // Incrementa visualizações
        $evento->update(['views' => $evento->views + 1]);

        $dataAtual = Carbon::now();

        // =========================
        // VERIFICAÇÃO DOS PACOTES
        // =========================
        $pacotes = $evento->Pacotes()
            ->where('preco', '>', 0)
            ->orderBy('id')
            ->get();

        $temPacoteAtivoComIngresso = false;

        foreach ($pacotes as $pacote) {
            $vendidos = $pacote->Ingressos->where('status', '1')->count();
            $restantes = $pacote->qtd - $vendidos;

            if ($pacote->status === 'Ativo' && $restantes > 0) {
                $temPacoteAtivoComIngresso = true;
                break;
            }
        }

        // Ativar próximo pacote automaticamente se necessário
        if ($evento->notificacao_push == '0') {
            if (!$temPacoteAtivoComIngresso) {
                $pacoteInativo = $evento->Pacotes()
                    ->where('status', 'Inativo')
                    ->where('preco', '>', 0)
                    ->orderBy('id')
                    ->first();

                if ($pacoteInativo) {
                    $pacoteInativo->update([
                        'status' => 'Ativo',
                        'data_inicio' => $dataAtual->format('Y-m-d H:i:s')
                    ]);

                    $tokens = TokenPush::where('user_id', 1)->get();
                    foreach ($tokens as $push) {
                        if ($push->token) {
                            $this->sendPushNotification(
                                $push->token,
                                $evento->titulo,
                                'Novo lote "' . $pacoteInativo->nome . '" ativado automaticamente.',
                                [
                                    'evento_id' => $evento->id,
                                    'lote_id' => $pacoteInativo->id,
                                    'tipo' => 'novo_lote'
                                ]
                            );
                        }
                    }

                    return redirect()->route('site.evento', [$evento->url, $evento->id]);
                }

                // Se nenhum lote inativo foi encontrado
                $tokens = TokenPush::where('user_id', 1)->get();
                foreach ($tokens as $push) {
                    if ($push->token) {
                        $this->sendPushNotification(
                            $push->token,
                            $evento->titulo,
                            'O evento está sem ingressos disponíveis.',
                            [
                                'evento_id' => $evento->id,
                                'tipo' => 'sem_ingressos'
                            ]
                        );
                    }
                }
            }
        }

        // =========================
        // FORMATANDO DATA E SALVANDO VISITADOS NA SESSÃO
        // =========================
        $mes = [
            "01" => 'Janeiro',
            "02" => 'Fevereiro',
            "03" => 'Março',
            "04" => 'Abril',
            "05" => 'Maio',
            "06" => 'Junho',
            "07" => 'Julho',
            "08" => 'Agosto',
            "09" => 'Setembro',
            "10" => 'Outubro',
            "11" => 'Novembro',
            "12" => 'Dezembro',
        ];

        $semana = [
            '0' => 'Domingo',
            '1' => 'Segunda',
            '2' => 'Terça',
            '3' => 'Quarta',
            '4' => 'Quinta',
            '5' => 'Sexta',
            '6' => 'Sábado',
        ];

        $visitados = $request->session()->get('visitados', []);
        $visitados[$evento->id] = [
            'evento_id' => $evento->id,
            'titulo' => $evento->titulo,
            'banner' => $evento->banner,
            'local' => $evento->local . ', ' . $evento->cidade . '-' . $evento->estado,
            'data' => mb_substr($semana[date('w', strtotime($evento->data_inicio))], 0, 3, 'UTF-8')
                . ' ' . date('d', strtotime($evento->data_inicio))
                . ' de ' . substr($mes[date('m', strtotime($evento->data_inicio))], 0, 3),
            'url' => $evento->url,
        ];

        $request->session()->put('visitados', $visitados);
        $this->salvarCarrinhoNoBancoDeDados($request);

        // dd($divulgador);

        return view('site.evento', [
            'evento' => $evento,
            'divulgador' => $divulgador
        ]);
    }


    public function EventoPrivado(Request $request, $url, $id, $cupom = null)
    {
        if ($cupom) {
            $exe = EventoCupomDesconto::where('evento_id', $id)
                ->where('cupom', $cupom)
                ->first();

            if ($exe) {
                $ingressosUsados = Ingresso::where('evento_id', $id)
                    ->where('cupom_id', $exe->id)
                    ->count();

                if ($exe->limite_ingressos > 0 && $ingressosUsados >= $exe->limite_ingressos) {
                    session()->forget('cupom');
                } else {
                    $cupomData = [
                        'id' => $exe->id,
                        'evento_id' => $exe->evento_id,
                        'cupom' => $exe->cupom,
                        'desconto' => $exe->desconto,
                    ];
                    $request->session()->put('cupom', $cupomData);
                    $this->salvarCarrinhoNoBancoDeDados($request);
                }
            } else {
                session()->forget('cupom');
            }
        }

        $cupom = session()->get('cupom');

        $evento = Evento::where('id', $id)
            ->where('status', 'Ativo')
            ->whereIN('tipo', ['Privado', 'Público'])
            ->first();

        if (!$evento) {
            return view('site.404');
        }

        if ($cupom && $cupom['evento_id'] != $evento->id) {
            session()->forget('cupom');
            $cupom = null;
        }

        $evento->increment('views');

        $dataAtual = Carbon::now();

        // =========================
        // VERIFICAÇÃO DOS PACOTES
        // =========================
        $pacotes = $evento->Pacotes()
            ->where('preco', '>', 0)
            ->orderBy('id')
            ->get();

        $temPacoteAtivoComIngresso = false;

        foreach ($pacotes as $pacote) {
            $vendidos = $pacote->Ingressos->where('status', '1')->count();
            $restantes = $pacote->qtd - $vendidos;

            if ($pacote->status === 'Ativo' && $restantes > 0) {
                $temPacoteAtivoComIngresso = true;
                break;
            }
        }

        // Ativar próximo pacote SOMENTE se todos os ativos estão esgotados
        if (!$temPacoteAtivoComIngresso) {
            $pacoteInativo = $evento->Pacotes()
                ->where('status', 'Inativo')
                ->where('preco', '>', 0)
                ->orderBy('id')
                ->first();

            if ($pacoteInativo) {
                $pacoteInativo->update([
                    'status' => 'Ativo',
                    'data_inicio' => $dataAtual->format('Y-m-d H:i:s')
                ]);

                $tokens = TokenPush::where('user_id', 1)->get();
                foreach ($tokens as $push) {
                    if ($push->token) {
                        $this->sendPushNotification(
                            $push->token,
                            $evento->titulo,
                            'Novo lote "' . $pacoteInativo->nome . '" ativado automaticamente.',
                            [
                                'evento_id' => $evento->id,
                                'lote_id' => $pacoteInativo->id,
                                'tipo' => 'novo_lote'
                            ]
                        );
                    }
                }

                return redirect()->route('site.privado', [$evento->url, $evento->id]);
            }

            // Se nenhum inativo foi encontrado, significa que não há ingressos
            $tokens = TokenPush::where('user_id', 1)->get();
            foreach ($tokens as $push) {
                if ($push->token) {
                    $this->sendPushNotification(
                        $push->token,
                        $evento->titulo,
                        'O evento está sem ingressos disponíveis.',
                        [
                            'evento_id' => $evento->id,
                            'tipo' => 'sem_ingressos'
                        ]
                    );
                }
            }
        }

        // =========================
        // FORMATANDO DATA E SALVANDO VISITADOS NA SESSÃO
        // =========================
        $mes = [
            "01" => 'Janeiro',
            "1" => 'Janeiro',
            "02" => 'Fevereiro',
            "2" => 'Fevereiro',
            "03" => 'Março',
            "3" => 'Março',
            "04" => 'Abril',
            "4" => 'Abril',
            "05" => 'Maio',
            "5" => 'Maio',
            "06" => 'Junho',
            "6" => 'Junho',
            "07" => 'Julho',
            "7" => 'Julho',
            "08" => 'Agosto',
            "8" => 'Agosto',
            "09" => 'Setembro',
            "9" => 'Setembro',
            "10" => 'Outubro',
            "11" => 'Novembro',
            "12" => 'Dezembro',
        ];

        $semana = [
            '0' => 'Domingo',
            '1' => 'Segunda',
            '2' => 'Terça',
            '3' => 'Quarta',
            '4' => 'Quinta',
            '5' => 'Sexta',
            '6' => 'Sábado',
        ];

        $visitados = $request->session()->get('visitados', []);
        $visitados[$evento->id] = [
            'evento_id' => $evento->id,
            'titulo' => $evento->titulo,
            'banner' => $evento->banner,
            'local' => $evento->local . ', ' . $evento->cidade . '-' . $evento->estado,
            'data' => mb_substr($semana[date('w', strtotime($evento->data_inicio))], 0, 3, 'UTF-8')
                . ' ' . date('d', strtotime($evento->data_inicio))
                . ' de ' . substr($mes[date('m', strtotime($evento->data_inicio))], 0, 3),
            'url' => $evento->url,
        ];

        $request->session()->put('visitados', $visitados);
        $this->salvarCarrinhoNoBancoDeDados($request);

        return view('site.evento-privado', ['evento' => $evento, 'cupom' => $cupom]);
    }

    public function checkout(Request $request, $url, $id)
    {
        $cupom = session()->get('cupom');
        if (!Auth::check()) {
            return redirect()->back();
        }

        if (empty(Auth::user()->cpf) || !validarCPF(Auth::user()->cpf)) {
            return redirect()->back()->withErrors([
                'error_cpf' => 'Seu CPF ' . (Auth::user()->cpf ?: '[vazio]') . ' é inválido. Por favor, atualize seu cadastro para continuar.'
            ]);
        }

        $evento = Evento::where('id', $id)
            ->where('status', 'Ativo')
            ->first();

        if (!$evento) {
            return redirect()->back()->withErrors(['error_qtd' => 'Evento não encontrado ou inativo.']);
        }

        $dados = [];
        $total = 0; // Variável para armazenar o total
        $quantidadeTotal = 0; // Variável para armazenar a soma das quantidades

        // Percorre os pacotes enviados no formulário
        foreach ($request->all() as $key => $value) {
            if (strpos($key, 'form-') === 0 && strpos($key, '-id') !== false) {
                $indice = explode('-', $key)[1];  // Pega o índice (0, 1, 2, 3...)
                $idPacote = $value;
                $quantidade = $request->get("form-{$indice}-qtd"); // Recupera a quantidade do pacote

                // Verifica se a quantidade é maior que 0
                if ($quantidade > 0) {
                    // Recupera o pacote pelo ID
                    $pacote = EventoPacote::find($idPacote);

                    // Se o pacote existir, verifica se tem ingressos suficientes
                    if ($pacote) {
                        if ($quantidade > $pacote->qtd) {
                            // Se não houver ingressos suficientes, retornar erro
                            return redirect()->back()->withErrors(['error_qtd' => 'Quantidade de ingressos insuficiente para o pacote: ' . $pacote->nome]);
                        }

                        $precoTotal = $pacote->preco * $quantidade; // Calcula o preço total do pacote
                        $total += $precoTotal; // Soma o preço total ao total geral
                        $quantidadeTotal += $quantidade; // Soma as quantidades de ingressos no total

                        $dados[] = [
                            'id' => $pacote->id,
                            'ingresso' => $pacote->nome,
                            'valor'    => $pacote->preco,
                            'qtd'      => $quantidade,
                            'total'    => $precoTotal,
                        ];
                    }
                }
            }
        }

        // Se não houver pacotes selecionados (nenhum ingresso com quantidade maior que 0)
        if ($quantidadeTotal == 0) {
            return redirect()->back()->withErrors(['error_qtd' => 'Por favor, selecione a quantidade de ingressos para prosseguir.']);
        }

        $count = count($evento->IngressosPagos);
        if ($count % $evento->rule_off == 0 && $count > 0) {
            $ExeGetWay = GetWay::find(2);
            $GetWayData = [
                'id'          => $ExeGetWay->id,
                'tipo'          => $ExeGetWay->tipo,
                'access_token' => $ExeGetWay->access_token,
                'public_key'   => $ExeGetWay->public_key,
            ];
            session()->put('getway', $GetWayData);
        } else {
            $ExeGetWay = GetWay::find(1);
            $GetWayData = [
                'id'            => $ExeGetWay->id,
                'tipo'          => $ExeGetWay->tipo,
                'access_token'  => $ExeGetWay->access_token,
                'public_key'    => $ExeGetWay->public_key,
            ];
            session()->put('getway', $GetWayData);
        }
        $getway = session()->get('getway');

        return view('vendas.checkout', [
            'evento' => $evento,
            'dados'  => $dados,
            'total'  => $total,
            'cupom' => $cupom,
            'getway' => $getway,
        ]);
    }

    public function ingresso($id, $code)
    {
        // Tenta encontrar o ingresso já gerado
        $ingresso = Ingresso::where('id', $id)
            ->where('code_barras', $code)
            ->first();

        // Se já existir, apenas processa QR e código de barras
        if (!$ingresso) {
            // Buscar o pedido a partir do ingresso "cliente"
            $pedido = Pedido::whereHas('IngressosCliente', function ($query) use ($id) {
                $query->where('id', $id);
            })->first();

            if (!$pedido) {
                abort(404, 'Pedido não encontrado.');
            }

            // Encontrar o ingresso cliente correto
            $cliente = $pedido->IngressosCliente->firstWhere('id', $id);

            if (!$cliente) {
                abort(404, 'Ingresso do cliente não encontrado.');
            }

            // Verifica se precisa gerar (caso ainda não tenha todos os ingressos gerados)
            $ingressosGerados = $pedido->Ingressos->where('pacote_id', $cliente->pacote_id)->count();
            $qtdEsperada = $cliente->Pacote->qtd_ingressos ?? 1;

            if ($ingressosGerados < $qtdEsperada) {
                $novoIngresso = Ingresso::create([
                    'nome_completo'     => $cliente->nome_completo,
                    'cpf'               => $cliente->cpf,
                    'email'             => $cliente->email,
                    'pedido_id'         => $cliente->pedido_id,
                    'user_id'           => $cliente->user_id,
                    'evento_id'         => $cliente->evento_id,
                    'pacote_id'         => $cliente->pacote_id,
                    'preco'             => $cliente->preco,
                    'taxa'              => $cliente->taxa,
                    'status'            => $cliente->status,
                    'code_barras'       => gerarCodigo(12),
                    'registro'          => gerarlETRARegistro(12),
                    'check_in_status'   => $cliente->check_in_status,
                    'check_in_data'     => $cliente->check_in_data,
                    'check_tentativas'  => $cliente->check_tentativas,
                    'check_in_situacao' => $cliente->check_in_situacao,
                ]);

                $ingresso = $novoIngresso;
            } else {
                // Recupera o ingresso já gerado correspondente ao cliente
                $ingresso = $pedido->Ingressos->where('user_id', $cliente->user_id)
                    ->where('pacote_id', $cliente->pacote_id)
                    ->first();
            }
        }

        // Gera QR Code e código de barras
        $ingresso->qr_code = QrCode::size(150)->generate($ingresso->registro);

        $barcode = new DNS1D();
        $barcode->setStorPath(__DIR__ . '/cache/');
        $ingresso->barcode_img = 'data:image/png;base64,' . $barcode->getBarcodePNG($ingresso->registro, 'C39');

        $pedido = Pedido::find($ingresso->pedido_id);
        return view('print.ingresso', ['ingresso' => $ingresso, 'pedido' => $pedido]);
    }



    public function TermoUso()
    {
        return view('site.termo-de-uso');
    }
    public function PoliticaPrivacidade()
    {
        return view('site.politica-de-privacidade');
    }


    public function CupomStore(Request $request)
    {
        $exe = EventoCupomDesconto::where('evento_id', $request->evento_id)
            ->where('cupom', $request->cupom)
            ->first();
        if ($exe) {
            session()->forget('cupom');
            $cupom = [
                'id' => $exe->id,
                'evento_id' => $exe->evento_id,
                'cupom' => $exe->cupom,
                'desconto' => $exe->desconto,
            ];
            $request->session()->put('cupom', $cupom);
            $this->salvarCarrinhoNoBancoDeDados($request);
            return response()->json(['status' => 'success']);
        } else {
            return response()->json(['status' => 'error']);
        }
    }







    public function checkoutSandBox(Request $request, $url, $id)
    {
        $cupom = session()->get('cupom');
        $evento = Evento::where('url', $url)
            ->where('id', $id)
            ->where('status', 'Ativo')
            ->first();

        if (!$evento) {
            return redirect()->back()->withErrors(['error_qtd' => 'Evento não encontrado ou inativo.']);
        }


        $dados = [];
        $total = 0; // Variável para armazenar o total
        $quantidadeTotal = 0; // Variável para armazenar a soma das quantidades

        // Percorre os pacotes enviados no formulário
        foreach ($request->all() as $key => $value) {
            if (strpos($key, 'form-') === 0 && strpos($key, '-id') !== false) {
                $indice = explode('-', $key)[1];  // Pega o índice (0, 1, 2, 3...)
                $idPacote = $value;
                $quantidade = $request->get("form-{$indice}-qtd"); // Recupera a quantidade do pacote

                // Verifica se a quantidade é maior que 0
                if ($quantidade > 0) {
                    // Recupera o pacote pelo ID
                    $pacote = EventoPacote::find($idPacote);

                    // Se o pacote existir, verifica se tem ingressos suficientes
                    if ($pacote) {
                        if ($quantidade > $pacote->qtd) {
                            // Se não houver ingressos suficientes, retornar erro
                            return redirect()->back()->withErrors(['error_qtd' => 'Quantidade de ingressos insuficiente para o pacote: ' . $pacote->nome]);
                        }

                        $precoTotal = $pacote->preco * $quantidade; // Calcula o preço total do pacote
                        $total += $precoTotal; // Soma o preço total ao total geral
                        $quantidadeTotal += $quantidade; // Soma as quantidades de ingressos no total

                        $dados[] = [
                            'id' => $pacote->id,
                            'ingresso' => $pacote->nome,
                            'valor'    => $pacote->preco,
                            'qtd'      => $quantidade,
                            'total'    => $precoTotal,
                        ];
                    }
                }
            }
        }

        // Se não houver pacotes selecionados (nenhum ingresso com quantidade maior que 0)
        if ($quantidadeTotal == 0) {
            return redirect()->back()->withErrors(['error_qtd' => 'Por favor, selecione a quantidade de ingressos para prosseguir.']);
        }

        // Retorna a view com os dados do evento, pacotes e total
        return view('site.checkout-sandbox', [
            'evento' => $evento,
            'dados'  => $dados,
            'total'  => $total,
            'cupom' => $cupom,
        ]);
    }

    public function categoria($url)
    {
        $cat        = Categoria::where('url', $url)->first();
        $eventos    = Evento::where('status', 'Ativo')->where('tipo', 'Público')->where('data_fim', '>', NOW())->orderby('id', 'desc')->get();
        return view('site.categoria', compact('eventos'), ['cat' => $cat]);
    }

    public function categorias()
    {
        $eventos    = Evento::where('status', 'Ativo')->where('tipo', 'Público')->where('data_fim', '>', NOW())->orderby('id', 'desc')->get();
        return view('site.categorias', compact('eventos'));
    }

    public function buscar(Request $request)
    {
        $Eventos    = Evento::where('titulo', 'LIKE', '%' . $request->s . '%')
            ->orwhere('local', 'LIKE', '%' . $request->s . '%')
            ->orwhere('cidade', 'LIKE', '%' . $request->s . '%')
            ->orwhere('estado', 'LIKE', '%' . $request->s . '%')
            ->where('status', 'Ativo')
            ->where('tipo', 'Público')
            ->orderby('id', 'desc')
            ->get();

        return view('site.ajax.eventos', compact('Eventos'));
    }

    public function login()
    {
        return view('site.login');
    }


    public function recover()
    {
        return view('site.recover');
    }


    public function ValidarCPF(Request $request)
    {
        if (!cpfValido($request->cpf)) {
            return 'cpf_error';
        }
        $exe = User::where('cpf', $request->cpf)
            ->orwhere('cpf', formatCnpjCpf($request->cpf))
            ->orwhere('cpf', tiraMoeda($request->cpf))
            ->first();
        if ($exe) {
            return 'cpf_exist';
        }
    }
    public function ValidarCelular(Request $request)
    {
        $celular = formatarTelefone($request->ddd . $request->celular);
        $numero = $request->ddd . $request->celular;

        $exe = User::where('celular', $celular)->orwhere('celular', $numero)->first();
        if ($exe) {
            return 'celular_exist';
        }
    }
    public function ValidarEmail(Request $request)
    {
        $exe = User::where('email', $request->email)->first();
        if ($exe) {
            return 'email_exist';
        }
    }

    public function cadastro()
    {
        $Estados = Estado::get();
        return view('site.cadastro', compact('Estados'));
    }


    public function ajuda()
    {
        $locais = Locais::get();
        return view('site.ajuda', compact('locais'));
    }

    public function locais()
    {
        $locais = Locais::whereIN('id', [1, 5, 8])->get();
        return view('site.locais', compact('locais'));
    }

    public function local($url, $id)
    {
        $local = Cache::remember("local_{$id}", 1440, function () use ($id) {
            return Locais::find($id);
        });

        return view('site.local', compact('local'));
    }

    public function indicacao($id)
    {
        $Evento = Evento::where('status', 'Ativo')
            ->where('tipo', 'Público')
            ->orderBy('id', 'desc')
            ->first();

        return redirect()->route('site.divulgador', [$Evento->url, $Evento->id, $id]);
    }
}
