<?php

namespace App\Http\Controllers;

use App\Models\Evento;
use App\Models\EventoCupomDesconto;
use App\Models\Ingresso;
use App\Models\Pedido;
use Hamcrest\Core\Every;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class RelatoriosController extends Controller
{
    public function index()
    {
        if (Auth::user()->nivel > '1') {
            $Eventos = Evento::where('produtor_id', Auth::user()->produtor_id)->get();
        } else {
            $Eventos = Evento::get();
        }
        return view('admin.relatorios.index', compact('Eventos'));
    }

    public function gerar(Request $request)
    {
        // 1) Validação
        $dados = $request->validate([
            'evento_id' => ['required', 'integer'],
            'inicio'    => ['nullable', 'date_format:Y-m-d'],
            'fim'       => ['nullable', 'date_format:Y-m-d', 'after_or_equal:inicio'],
            'status'    => ['nullable', 'in:Pagos,Cancelados,Todos']
        ]);

        $eventoId = (int) $dados['evento_id'];

        // 2) Datas
        $inicio = !empty($dados['inicio'])
            ? Carbon::createFromFormat('Y-m-d', $dados['inicio'])->startOfDay()
            : null;

        $fim = !empty($dados['fim'])
            ? Carbon::createFromFormat('Y-m-d', $dados['fim'])->endOfDay()
            : null;

        // 3) Status
        $statusMap = [
            'Pagos'      => '1',
            'Cancelados' => '0',
            'Todos'      => null,
            null         => null,
            ''           => null,
        ];
        $statusFiltro = $statusMap[$dados['status'] ?? null];

        // 4) Atualização de ingressos a partir dos pedidos
        try {
            DB::transaction(function () {
                Pedido::query()
                    ->whereHas('Ingressos', fn($q) => $q->where('check_in_situacao', '0'))
                    ->with([
                        'Ingressos:id,pedido_id,evento_id,pacote_id,preco,valor_final,pdv,cupom,cupom_id,taxa,desconto',
                        'Ingressos.Pacote:id,preco,qtd_ingressos',
                        'Ingressos.Evento:id,taxa_servico',
                    ])
                    ->orderBy('id')
                    ->chunkById(200, function ($pedidos) {
                        foreach ($pedidos as $pedido) {
                            foreach ($pedido->Ingressos as $ingresso) {
                                $pacote = $ingresso->Pacote;
                                if (!$pacote) continue;

                                $precoPacote = is_numeric($pacote->preco)
                                    ? (float) $pacote->preco
                                    : (float) str_replace(',', '.', $pacote->preco);

                                $qtdIng = (int) ($pacote->qtd_ingressos ?? 1);
                                if ($qtdIng < 1) $qtdIng = 1;

                                $valorUnitBase = $qtdIng > 1
                                    ? round($precoPacote / $qtdIng, 2)
                                    : $precoPacote;

                                // Se tiver cupom
                                if ($ingresso->cupom_id) {
                                    $cupom = EventoCupomDesconto::find($ingresso->cupom_id);
                                    $pct   = (float) ($cupom->desconto ?? 0);

                                    $desconto = $pct > 0
                                        ? round($valorUnitBase * ($pct / 100), 2)
                                        : 0;

                                    $precoComDesconto = $valorUnitBase - $desconto;

                                    // taxa sobre o preço com desconto
                                    $taxa = porcentagem(
                                        $ingresso->Evento->taxa_servico,
                                        $precoComDesconto
                                    );

                                    $ingresso->update([
                                        'desconto'     => $pct,
                                        'preco'        => $precoComDesconto,
                                        'taxa'         => $taxa,
                                        'valor_final'  => $precoComDesconto + $taxa,
                                        'pdv'          => $pedido->pdv,
                                        'cupom'        => $pedido->cupom,     // já é o nome
                                        'cupom_id'     => $pedido->cupom_id,
                                    ]);
                                } else {
                                    // Sem cupom
                                    $taxaUnit = porcentagem(
                                        $ingresso->Evento->taxa_servico,
                                        $valorUnitBase
                                    );

                                    $ingresso->update([
                                        'desconto'     => 0,
                                        'preco'        => $valorUnitBase,
                                        'taxa'         => $taxaUnit,
                                        'valor_final'  => $valorUnitBase + $taxaUnit,
                                        'pdv'          => $pedido->pdv,
                                        'cupom'        => $pedido->cupom,
                                        'cupom_id'     => $pedido->cupom_id,
                                    ]);
                                }
                            }
                        }
                    });
            }, 3);
        } catch (\Throwable $e) {
            Log::error('RelatoriosController@gerar: erro ao atualizar ingressos pelos pedidos', [
                'message' => $e->getMessage(),
            ]);
        }

        // 5) Query de ingressos filtrados
        $ingressosBuilder = Ingresso::with(['Pacote', 'Cupom'])
            ->where('evento_id', $eventoId)
            ->where('check_in_situacao', '0');

        if (!is_null($statusFiltro)) {
            $ingressosBuilder->where('status', $statusFiltro);
        }

        if ($inicio && $fim) {
            $ingressosBuilder->whereBetween('created_at', [$inicio, $fim]);
        }

        $ingressosCollection = $ingressosBuilder->get();

        // 6) Totais
        $toFloat = fn($v) => (float) ($v ?? 0);

        $total_geral_ingressos = $ingressosCollection->count();
        $total_geral_vendas    = $ingressosCollection->sum(fn($i) => $toFloat($i->preco));

        $vendasManuais = $ingressosCollection
            ->filter(fn($i) => ($i->pdv ?? null) === 'Manual')
            ->sum(fn($i) => $toFloat($i->preco));

        $vendasSite = $ingressosCollection
            ->filter(fn($i) => ($i->pdv ?? null) === 'Site')
            ->sum(fn($i) => $toFloat($i->preco));

        // 7) Formatador
        $formatMoney = function ($valor) {
            if (function_exists('moeda')) return moeda($valor);
            return number_format((float) $valor, 2, ',', '.');
        };

        // 8) Agrupamento por pacote+preço+cupom
        $ingressos = $ingressosCollection
            ->groupBy(function ($i) {
                $preco = number_format((float) ($i->preco ?? 0), 2, '.', '');
                $cupomKey = $i->cupom_id ?? (($i->cupom ?? '') !== '' ? $i->cupom : 'SEM_CUPOM');
                $pacoteKey = $i->pacote_id ?? '0';
                return $pacoteKey . '|' . $preco . '|' . $cupomKey;
            })
            ->map(function ($group) use ($formatMoney, $toFloat) {
                $first  = $group->first();
                $pacote = optional($first->Pacote);
                $cupom  = optional($first->Cupom);

                $totalVendas = $group->sum(fn($x) => $toFloat($x->preco));

                return [
                    'pdv'             => $first->pdv ?? 'N/A',
                    'pacote_id'       => $first->pacote_id ?? null,
                    'pacote_nome'     => $pacote->nome ?? 'N/A',
                    'cupom_id'        => $first->cupom_id ?? null,
                    'cupom'           => $cupom->cupom ?? ($first->cupom ?: 'SEM CUPOM'), // nome do cupom
                    'desconto'        => $cupom->desconto ?? 0,
                    'preco_ingresso'  => $formatMoney($pacote->preco ?? 0),
                    'preco_venda'     => $formatMoney($first->preco ?? 0),
                    'total'           => $group->count(),
                    'total_vendas'    => $formatMoney($totalVendas),
                ];
            })
            ->values();

        // 9) Resumo por cupom
        $porCupom = $ingressosCollection
            ->groupBy(function ($i) {
                return $i->cupom_id ?? (($i->cupom ?? '') !== '' ? $i->cupom : 'SEM_CUPOM');
            })
            ->map(function ($group) use ($formatMoney, $toFloat) {
                $first = $group->first();
                $cupom = optional($first->Cupom);

                $totalVendas = $group->sum(fn($x) => $toFloat($x->preco));

                return [
                    'cupom_id'     => $first->cupom_id ?? null,
                    'cupom'        => $cupom->cupom ?? ($first->cupom ?: 'SEM CUPOM'),
                    'desconto'     => $cupom->desconto ?? 0,
                    'qtd'          => $group->count(),
                    'total_vendas' => $formatMoney($totalVendas),
                ];
            })
            ->values();

        // 10) Retorno
        return view('admin.relatorios.ajax.tabela', compact('ingressos', 'porCupom'), [
            'total_geral_ingressos' => $total_geral_ingressos,
            'total_geral_vendas'    => $total_geral_vendas,
            'total_vendas_manuais'  => $vendasManuais,
            'total_vendas_site'     => $vendasSite,
        ]);
    }




    public function mensal()
    {
        $eventos = [];
        $GetEventos = Evento::get();
        return view('admin.relatorios.mensal', compact('eventos', 'GetEventos'));
    }

    public function GerarMensal(Request $request)
    {
        $GetEventos = Evento::get();
        $inicio = $request->input('inicio');
        $fim = $request->input('fim');


        // dd($inicio, $fim, $request->all());

        $pedidos = Pedido::query()
            ->where('status', 'Aprovado')
            ->where('check_in_situacao', '0');

        // Se houver período, aplica o filtro
        if ($inicio && $fim) {
            $pedidos->whereBetween('created_at', [$inicio, $fim]);
        }

        // Se houver evento_id, aplica o filtro por evento
        if ($request->filled('evento_id')) {
            $pedidos->where('evento_id', $request->evento_id);
        }

        $pedidos = $pedidos->get();

        // Agrupar por evento
        $eventos = $pedidos->groupBy('evento_id')->map(function ($pedidosDoEvento, $eventoId) {
            $evento = Evento::find($eventoId);

            $vendasOnline = $pedidosDoEvento->where('pdv', 'Site')->sum('sub_total');
            $vendasManuais = $pedidosDoEvento->where('pdv', 'Manual')->sum('total');

            $taxaClienteSite = porcentagem(
                $pedidosDoEvento->where('taxa_origen', 'Cliente')->where('pdv', 'Site')->sum('sub_total'),
                env('TAXA')
            );

            $taxaEmpresa = porcentagem(
                $pedidosDoEvento->where('taxa_origen', 'Empresa')->where('pdv', 'Site')->sum('total'),
                env('TAXA')
            );

            $taxaVendaManual = porcentagem(
                $pedidosDoEvento->where('pdv', 'Manual')->sum('total'),
                env('TAXA_VENDA_MANUAL')
            );

            $totalTaxas   = $taxaClienteSite + $taxaEmpresa + $taxaVendaManual;
            $totalVendas  = $pedidosDoEvento->sum('sub_total');
            $totalRepasse = $vendasOnline - ($taxaEmpresa + $taxaVendaManual);

            return [
                'evento'              => $evento,
                'vendas_online'       => $vendasOnline,
                'vendas_manuais'      => $vendasManuais,
                'taxa_cliente_site'   => $taxaClienteSite,
                'taxa_empresa'        => $taxaEmpresa,
                'taxa_venda_manual'   => $taxaVendaManual,
                'total_taxas'         => $totalTaxas,
                'total_vendas'        => $totalVendas + $totalTaxas,
                'total_repasse'       => $totalRepasse,
            ];
        });

        return view('admin.relatorios.mensal', compact('GetEventos'), [
            'pedidos' => $pedidos,
            'eventos' => $eventos,
            'inicio' => $inicio,
            'fim' => $fim,
        ]);
    }

    public function GerarTotalPacotes(Request $request)
    {
        $dados = $request->validate([
            'evento_id' => ['required', 'integer'],
            'inicio'    => ['nullable', 'date_format:Y-m-d'],
            'fim'       => ['nullable', 'date_format:Y-m-d', 'after_or_equal:inicio'],
            'pacotes'   => ['nullable', 'array'],
            'pacotes.*' => ['integer'],
        ]);

        $eventoId = (int) $dados['evento_id'];
        $pacotesSelecionados = $dados['pacotes'] ?? [];

        $inicio = !empty($dados['inicio'])
            ? Carbon::createFromFormat('Y-m-d', $dados['inicio'])->startOfDay()
            : null;

        $fim = !empty($dados['fim'])
            ? Carbon::createFromFormat('Y-m-d', $dados['fim'])->endOfDay()
            : null;

        $ingressosBuilder = Ingresso::with('Pacote')
            ->where('evento_id', $eventoId)
            ->where('check_in_situacao', '0')
            ->where('status', '1')
            ->whereHas('Pedido', function ($q) {
                $q->where('status', 'Aprovado');
            });

        if ($inicio && $fim) {
            $ingressosBuilder->whereBetween('created_at', [$inicio, $fim]);
        }

        if (!empty($pacotesSelecionados)) {
            $ingressosBuilder->whereIn('pacote_id', $pacotesSelecionados);
        }

        $ingressosCollection = $ingressosBuilder
            ->orderBy('created_at', 'desc')
            ->get();

        $formatMoney = function ($valor) {
            if (function_exists('moeda')) return moeda($valor);
            return number_format((float) $valor, 2, ',', '.');
        };

        $totaisPorData = $ingressosCollection
            ->groupBy(function ($ingresso) {
                // chave como data ISO (ordenável corretamente)
                return Carbon::parse($ingresso->created_at)->format('Y-m-d');
            })
            ->sortKeysDesc()
            ->mapWithKeys(function ($ingressosDoDia, $dataISO) use ($formatMoney) {
                // exibição em formato BR
                $dataFormatada = Carbon::parse($dataISO)->format('d/m/Y');

                $pacotes = $ingressosDoDia->groupBy('pacote_id')
                    ->map(function ($group) use ($formatMoney) {
                        $pacote = optional($group->first()->Pacote);

                        $vendas = $group->pluck('pedido_id')->unique()->count();
                        $ingressos = $group->count();
                        $totalReais = $formatMoney($group->sum('preco'));

                        return [
                            'id'          => $pacote->id ?? 'N/A',
                            'pacote_nome' => $pacote->nome ?? 'N/A',
                            'pacote_preco' => $pacote->preco ?? 'N/A',
                            'vendas'      => $vendas,
                            'ingressos'   => $ingressos,
                            'total_reais' => $totalReais,
                        ];
                    })
                    ->sortByDesc('ingressos') // pacotes mais vendidos primeiro
                    ->values();

                return [$dataFormatada => $pacotes];
            });

        $totalIngressos = $ingressosCollection->count();
        $totalVendas    = $ingressosCollection->pluck('pedido_id')->unique()->count();
        $totalReais     = $formatMoney($ingressosCollection->sum('preco'));

        return view('admin.relatorios.ajax.total-por-pacotes', [
            'totaisPorData'   => $totaisPorData,
            'totalIngressos'  => $totalIngressos,
            'totalVendas'     => $totalVendas,
            'totalReais'      => $totalReais,
        ]);
    }
}
