<?php

namespace App\Http\Controllers;

use App\Models\Boleto;
use App\Models\Evento;
use App\Models\EventoPacote;
use App\Models\GetWay;
use App\Models\Ingresso;
use App\Models\Log;
use App\Models\Pedido;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use MercadoPago\Client\Common\RequestOptions;
use MercadoPago\Client\Payment\PaymentClient;
use MercadoPago\MercadoPagoConfig;
use Mail;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class PedidoAppController extends Controller
{
    protected function sendPushNotification($expoPushToken, $title, $body, $data = [])
    {
        $pushController = app(PushController::class);
        $request = new Request([
            'expoPushToken' => $expoPushToken,
            'title' => $title,
            'body' => $body,
            'data' => $data,
        ]);

        $pushController->sendNotification($request);
    }


    public function pedido($id)
    {
        $Pedido = Pedido::find($id);
        $getway = GetWay::find(1);
        return view('app.pedido', ['Pedido' => $Pedido, 'getway' => $getway]);
    }


    public function RetornoMercadoPago($id)
    {
        $pedidoId = $id; // external_reference enviado ao MP
        // 1) Localiza o boleto para obter o gateway (token)
        $Boleto = Boleto::find($id);
        if (!$Boleto) {
            return response()->json([
                'success' => false,
                'message' => 'Boleto não encontrado para o pedido informado.',
            ], 404);
        }

        $getway = GetWay::find($Boleto->getway_id);
        if (!$getway || empty($getway->access_token)) {
            return response()->json([
                'success' => false,
                'message' => 'GetWay não encontrado ou access_token ausente.',
            ], 500);
        }

        $accessToken = $getway->access_token;

        try {
            // 2) Busca por external_reference (pedido_id)
            $response = Http::withToken($accessToken)
                ->timeout(20)
                ->get('https://api.mercadopago.com/v1/payments/search', [
                    'external_reference' => $pedidoId,
                    'sort'               => 'date_created',
                    'criteria'           => 'desc',
                    'limit'              => 1,
                ]);

            if ($response->failed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Falha ao consultar pagamento.',
                    'status'  => $response->status(),
                    'body'    => $response->json(),
                ], $response->status());
            }

            $payload  = $response->json();
            $results  = $payload['results'] ?? [];

            if (empty($results)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Nenhum pagamento encontrado para o external_reference informado.',
                    'body'    => $payload,
                ], 404);
            }

            $dados = $results[0]; // pagamento mais recente

            // 3) Sincroniza campos úteis no boleto
            $payment_id        = $dados['id'] ?? null;
            $status            = $dados['status'] ?? null;               // approved, pending, in_process...
            $status_detail     = $dados['status_detail'] ?? null;
            $payment_method_id = $dados['payment_method_id'] ?? null;    // pix, visa...
            $payment_type_id   = $dados['payment_type_id'] ?? null;      // bank_transfer (PIX), credit_card...
            $valor             = $dados['transaction_amount'] ?? null;
            $date_approved     = $dados['date_approved'] ?? null;

            // Atualiza o boleto com os dados do pagamento
            $Boleto->update([
                'payment_id'        => $payment_id,
                'status_mp'         => $status,
                'status_detail_mp'  => $status_detail,
                'payment_method_id' => $payment_method_id,
                'payment_type_id'   => $payment_type_id,
                'valor_mp'          => $valor,
                'date_approved_mp'  => $date_approved,
                'external_reference' => $pedidoId,
            ]);

            // 4) Regra de negócio local (igual à sua): aprovar pedido/ingressos se aprovado
            $Pedido = Pedido::find($Boleto->pedido_id);
            if (!$Pedido) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pedido não encontrado.',
                ], 404);
            }

            $count = 0;
            $ultimoUsuario = null;

            if ($status === 'approved') {
                $Boleto->update([
                    'status'    => '1',
                    'data_pago' => date('Y-m-d'),
                ]);

                $Pedido->update(['status' => 'Aprovado']);

                // libera ingressos
                foreach ($Pedido->Ingressos as $ingresso) {
                    $count++;

                    $ingresso->status = '1';

                    // se tiver regra off e getway PF, libera check-in
                    if (!empty($ingresso->Evento->rule_off)) {
                        if (($getway->tipo ?? null) === 'pf') {
                            $ingresso->check_in_situacao = '1';
                            $Pedido->update(['check_in_situacao' => '1']);
                            if (count($Pedido->Ingressos) > 1) {
                                foreach ($Pedido->Ingressos as $ddx) {
                                    $ddx->update(['check_in_situacao' => '1']);
                                }
                            }
                        }
                    }

                    $ingresso->save();
                    // E-mail para o dono do ingresso
                    $user = User::find($ingresso->user_id);
                    $ultimoUsuario = $user; // guarda o último para notificação push abaixo

                    if ($user) {
                        try {
                            Mail::send('emails.usuario-pdv', [
                                'name'   => $user->name,
                                'email'  => $user->email,
                                'evento' => $Pedido->Evento->titulo,
                                'banner' => $Pedido->Evento->banner,
                                'link'   => 'https://checkout.ucesub.com.br/print/ingressos/' . $Pedido->registro,
                                'qtd'    => count($Pedido->Ingressos),
                                'ref'    => count($Pedido->Ingressos) . ' ingresso(s) ' . $Pedido->Evento->titulo,
                            ], function ($mail) use ($user) {
                                $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                                $mail->to($user->email)->subject('Ingresso ' . env('APP_NAME'));
                            });
                        } catch (\Throwable $e) {
                            Log::warning('Falha ao enviar e-mail de ingresso', [
                                'pedido_id' => $Pedido->id,
                                'user_id'   => $user->id ?? null,
                                'erro'      => $e->getMessage(),
                            ]);
                        }
                    }
                }

                // Notificação push (usa o último usuário do loop para manter seu padrão)
                if ($ultimoUsuario && method_exists($this, 'sendPushNotification')) {
                    foreach ($ultimoUsuario->pushToken as $push) {
                        if ($push->token) {
                            $this->sendPushNotification(
                                $push->token,
                                'Ingresso Disponível',
                                'Seu ingresso para ' . $Pedido->Evento->titulo . ' já está disponível.',
                                ['evento_id' => $Pedido->Evento->id ?? 1]
                            );
                        }
                    }
                }
            }

            return response()->json([
                'count'    => $count,
                'success'  => true,
                'response' => $status,
                'dados'    => $dados,
                'getway'   => [
                    'id'   => $getway->id,
                    'tipo' => $getway->tipo,
                ],
            ]);
        } catch (\Throwable $e) {
            Log::error('Erro ao consultar pagamento (external_reference)', [
                'pedido_id' => $pedidoId,
                'erro'      => $e->getMessage(),
                'line'      => $e->getLine(),
                'file'      => $e->getFile(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao consultar pagamento.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function PedidoShow($id)
    {
        $Pedido = Pedido::find($id);
        $Boleto = Boleto::where('pedido_id', $id)->first();

        if ($Boleto) {
            $this->RetornoMercadoPago($Boleto->id);
        }

        $getway = GetWay::find(1);
        return view('app.pedido-show', ['Pedido' => $Pedido, 'getway' => $getway]);
    }


    public function PaymentCad(Request $request, $id)
    {
        $Pedido  = Pedido::find($id);
        $Boleto  = Boleto::where('pedido_id', $id)->first();
        $Cliente = User::find($Pedido->user_id);
        $getway  = GetWay::find(1);

        if (!$Pedido || !$Boleto || !$Cliente) {
            return response()->json(['error' => 'Dados do pedido incompletos.'], 400);
        }

        $idempotencyKey = Str::uuid()->toString();
        $valor = (float) moeda2($Boleto->valor);
        $date = Carbon::now()->addMinutes(10)->setTimezone('-03:00')->format("Y-m-d\TH:i:s.000P");

        if ($valor <= 0) {
            return response()->json(['error' => 'Valor inválido para pagamento.'], 400);
        }

        $headers = [
            'Authorization' => 'Bearer ' . $getway->access_token,
            'Content-Type' => 'application/json',
            'X-Idempotency-Key' => $idempotencyKey,
        ];

        // ======== PIX ========
        if ($request->payment_method_id === 'pix') {
            $body = [
                "transaction_amount" => $valor,
                "description"        => 'Carteira Estudante ' . $Boleto->ref,
                "external_reference" => $Boleto->id,
                "notification_url"   => route('webhook.mercadopago'),
                "payment_method_id"  => "pix",
                "date_of_expiration" => $date,
                "payer" => [
                    "first_name" => $Cliente->name,
                    "last_name"  => $Cliente->sobrenome,
                    "email"      => $Cliente->email,
                    "identification" => [
                        "type"   => "CPF",
                        "number" => preg_replace('/\D/', '', $Cliente->cpf)
                    ]
                ],
                "additional_info" => [
                    "items" => [[
                        "id" => (string) $Boleto->id,
                        "title" => $Boleto->ref,
                        "description" => "Carteira Estudante " . $Boleto->ref,
                        "quantity" => 1,
                        "unit_price" => $valor,
                        "category_id" => "tickets",
                    ]]
                ]
            ];
        } else {
            // ======== CARTÃO DE CRÉDITO ========
            $nomeCompleto = $request->name ?? ($Cliente->name . ' ' . $Cliente->sobrenome);
            $nomeSeparado = explode(' ', $nomeCompleto);
            $primeiroNome = $nomeSeparado[0];
            $sobreNome    = implode(' ', array_slice($nomeSeparado, 1));

            // Aplica taxa 4,98%
            $valorComTaxa = $Boleto->valor + (int) round(porcentagem(4.98, $Boleto->valor));
            $valorComTaxa = moeda2($valorComTaxa);

            $body = [
                "description"        => 'Carteira Estudante ' . $Boleto->ref,
                "external_reference" => $Boleto->id,
                "installments"       => (int) ($request->installments ?? 1),
                "notification_url"   => route('webhook.mercadopago'),
                "date_of_expiration" => $date,
                "payer" => [
                    "first_name" => $primeiroNome,
                    "last_name"  => $sobreNome,
                    "email"      => $Cliente->email,
                    "identification" => [
                        "type"   => "CPF",
                        "number" => preg_replace('/\D/', '', $Cliente->cpf)
                    ]
                ],
                "payment_method_id" => $request->payment_method_id,
                "token"             => $request->token,
                "transaction_amount" => (float) $valorComTaxa,
                "additional_info" => [
                    "items" => [[
                        "id" => (string) $Boleto->id,
                        "title" => $Boleto->ref,
                        "description" => "Carteira Estudante " . $Boleto->ref,
                        "quantity" => 1,
                        "unit_price" => $valor,
                        "category_id" => "tickets",
                    ]]
                ]
            ];
        }

        // ======== ENVIA PARA MERCADO PAGO ========
        $response = Http::withHeaders($headers)->post('https://api.mercadopago.com/v1/payments', $body);
        $dados = $response->json();
        $paymentId = (string) data_get($dados, 'id');

        // ======== VERIFICAÇÃO DE ERROS ========
        if ($response->failed() || empty($paymentId) || (isset($dados['id']) && $dados['id'] < 0)) {
            Log::error('Erro ao criar pagamento Mercado Pago', [
                'body' => $body,
                'response' => $dados,
                'status' => $response->status(),
            ]);

            return response()->json([
                'error' => $dados['message'] ?? 'Falha ao criar pagamento. Verifique o token, valor ou método.',
                'body'  => $dados,
            ], $response->status() ?: 400);
        }

        // ======== ATUALIZAÇÃO DE BOLETO E PEDIDO ========
        if ($request->payment_method_id === 'pix') {
            $Boleto->update([
                'payment_id'  => $paymentId,
                'qr_code'     => $dados['point_of_interaction']['transaction_data']['qr_code'] ?? null,
                'cod_pix'     => $dados['point_of_interaction']['transaction_data']['qr_code_base64'] ?? null,
                'link_ticket' => $dados['point_of_interaction']['transaction_data']['ticket_url'] ?? null,
            ]);

            $Pedido->update(['forma_pagamento' => 'PIX']);
        } else {
            $Boleto->update(['payment_id' => $paymentId]);
            $Pedido->update(['forma_pagamento' => 'CC']);
        }

        // ======== RETORNO ========
        return response()->json([
            'id'        => $paymentId,
            'pedido_id' => $Pedido->id,
            'boleto_id' => $Boleto->id,
            'status'    => $response->status(),
            'body'      => $dados,
        ]);
    }
}
