<?php

namespace App\Http\Controllers;

use App\Models\Boleto;
use App\Models\Evento;
use App\Models\EventoPacote;
use App\Models\GetWay;
use App\Models\Ingresso;
use App\Models\Log;
use App\Models\MercadoPago;
use App\Models\Pedido;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use MercadoPago\Client\Common\RequestOptions;
use MercadoPago\Client\Payment\PaymentClient;
use MercadoPago\MercadoPagoConfig;
use Mail;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class PagamentoController extends Controller
{

    protected function sendPushNotification($expoPushToken, $title, $body, $data = [])
    {
        $pushController = app(PushController::class);
        $request = new Request([
            'expoPushToken' => $expoPushToken,
            'title' => $title,
            'body' => $body,
            'data' => $data,
        ]);

        $pushController->sendNotification($request);
    }

    public function PaymentCad(Request $request)
    {
        $getway = session()->get('getway');
        $cupom  = session()->get('cupom');
        $cart           = [];
        $totalPedido    = 0;

        // dd($getway);

        // Itera sobre os dados do formulário
        foreach ($request->all() as $key => $value) {
            // Identifica os índices relacionados a cada formulário
            if (preg_match('/form-(\d+)-ingresso/', $key, $matches)) {
                $index = $matches[1];

                // Garante que são arrays, mesmo que venham como string
                $ingressosIds   = (array) $request->input("form-{$index}-ingresso");
                $nomesCompletos = (array) $request->input("form-{$index}-nome_completo");
                $emails         = (array) $request->input("form-{$index}-email");

                foreach ($ingressosIds as $i => $ingressoId) {
                    if (!$ingressoId) continue;

                    $nomeCompleto = $nomesCompletos[$i] ?? Auth::user()->name . ' ' . Auth::user()->sobrenome;
                    $email        = $emails[$i] ?? Auth::user()->email;

                    $eventoPacote = EventoPacote::find($ingressoId);
                    $precoPacote  = $eventoPacote ? $eventoPacote->preco : 0;

                    $cart[] = [
                        'nome_completo' => $nomeCompleto,
                        'email'         => $email,
                        'ingresso_id'   => $ingressoId,
                        'preco'         => $precoPacote
                    ];

                    if ($cupom) {
                        $desconto    = $cupom['desconto'] ?? 0;
                        $precoPacote -= $precoPacote * ($desconto / 100);
                    }

                    $totalPedido += $precoPacote;
                }
            }
        }

        $Evento = Evento::find($request->evento_id);
        // $taxaProcessamento = $request->token ? 0 : 0; // Taxa de processamento fixa
        $taxaProcessamento = $request->token ? 4.98 : 0; // Taxa de processamento fixa
        $juros = [
            1 => 0,
            2 => 2.53,
            3 => 4.62,
            4 => 6.69,
            5 => 8.66,
            6 => 9.96,
            7 => 11.24,
            8 => 12.50,
            9 => 13.73,
            10 => 14.93,
        ];
        $total          = $totalPedido + porcentagem($Evento->taxa_servico, $totalPedido);
        $totalJuros     = $taxaProcessamento + $juros[$request->installments ?? 1];

        $dadosCarrinho = [
            'qtd_ingressos'      => count($cart),
            'carrinho'      => $cart,
            'juros'         => $juros[$request->installments ?? 1],
            'taxa_servico'  => porcentagem(env('TAXA'), $totalPedido),
            'taxa'          => $taxaProcessamento,
            'TxJuros'       => $totalJuros,
            'totalPedido'   => $totalPedido,
            'total'         => $total + porcentagem($totalJuros, $total),
        ];

        $nomeCompleto = $request->name;
        $nomeSeparado = explode(' ', $nomeCompleto);
        $primeiroNome = $nomeSeparado[0];
        $sobreNome      = implode(' ', array_slice($nomeSeparado, 1));

        $user = User::find(Auth::user()->id);

        do {
            $registro = gerarLetraRegistro(12);
        } while (Pedido::where('registro', $registro)->exists());

        $pedido = Pedido::create([
            'evento_id'         => $request->evento_id,
            'user_id'           => Auth::user()->id,
            'taxa_origen'       => $Evento->taxa_servico == 0 ? 'Empresa' : 'Cliente',
            'pdv'               => 'Site',
            'qtd_pacotes'       => count($cart),
            'total'             => $dadosCarrinho['total'],
            'sub_total'         => $dadosCarrinho['totalPedido'],
            'taxa'              => $dadosCarrinho['taxa_servico'],
            'forma_pagamento'   => $request->token ? 'CC' : 'PIX',
            'status'            => 'Pendente',
            'registro'          => $registro,
            'cupom'             => $cupom ? $cupom['cupom'] : '',
            'cupom_id'          => $cupom ? $cupom['id'] : null
        ]);

        $Boleto = Boleto::create([
            'getway_id'   => $getway['id'],
            'pedido_id'   => $pedido->id,
            'user_id'     => Auth::user()->id,
            'status'      => '0',
            'valor'       => $dadosCarrinho['total'],
            'ref'         => count($cart) . ' ' . (count($cart) > 1 ? 'ingressos' : 'ingresso') . ' ' . $Evento->titulo,
            'vencimento'  => now()->addDays(2),
            'taxa_lopak'  => 50,
        ]);

        foreach ($cart as $item) {
            $pacote = EventoPacote::find($item['ingresso_id']);

            $qtdIngressos = $pacote->qtd_ingressos > 0 ? $pacote->qtd_ingressos : 1;

            // Corrige a divisão: pega o preço do pacote individual do item
            $precoPacote = $item['preco'];
            $precoUnitario = $precoPacote / $qtdIngressos;
            $taxaUnitaria  = porcentagem(env('TAXA'), $precoUnitario);
            $valorFinal    = $precoUnitario + $taxaUnitaria;

            for ($i = 0; $i < $qtdIngressos; $i++) {
                do {
                    $registro = gerarLetraRegistro(12);
                } while (Ingresso::where('registro', $registro)->exists());

                do {
                    $code_barras = gerarCodigo(12);
                } while (Ingresso::where('code_barras', $code_barras)->exists());

                $dados = [
                    'pedido_id'         => $pedido->id,
                    'user_id'           => $user->id,
                    'evento_id'         => $request->evento_id,
                    'pacote_id'         => $item['ingresso_id'],
                    'pdv'               => 'Site',
                    'status'            => '0',
                    'check_in_status'   => '0',
                    'check_tentativas'  => '0',
                    'code_barras'       => $code_barras,
                    'registro'          => $registro,
                    'num_acessos'       => 0,
                    'nome_completo'     => $item['nome_completo'],
                    'email'             => $item['email'],
                    'preco'             => $precoUnitario,
                    'taxa'              => $taxaUnitaria,
                    'valor_final'       => $valorFinal,
                    'cupom'             => $cupom ? $cupom['cupom'] : '',
                    'cupom_id'          => $cupom ? $cupom['id'] : null
                ];

                Ingresso::create($dados);
            }
        }

        $date = Carbon::now()
            ->addMinutes(10)
            ->setTimezone('-03:00') // força fuso horário correto
            ->format("Y-m-d\TH:i:s.000P"); // inclui milissegundos .000

        if ($request->token) {
            $idempotencyKey = Str::uuid()->toString();
            $body = [
                "description" => count($cart) . ' ' . (count($cart) > 1 ? 'ingressos' : 'ingresso') . ' ' . $Evento->titulo,
                "external_reference" => $Boleto->id,
                "installments" => isset($request->installments) ? (int) $request->installments : 1,
                "notification_url" => 'https://checkout.ucesub.com.br/retorno-mp',
                "date_of_expiration" => $date,
                "payer" => [
                    "first_name" => $primeiroNome,
                    "last_name" => $sobreNome,
                    "email" => $request->email,
                    "identification" => [
                        "type" => "CPF",
                        "number" => preg_replace('/[^0-9]/', '', $request->cpf ?? Auth::user()->cpf)
                    ]
                ],
                "payment_method_id" => $request->payment_method_id,
                "token" => $request->token,
                "transaction_amount" =>  (float) moeda2($Boleto->valor),
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $getway['access_token'],
                'Content-Type' => 'application/json',
                'X-Idempotency-Key' => $idempotencyKey,
            ])->post('https://api.mercadopago.com/v1/payments', $body);

            // dd($response->json());

            $paymentId = (string) data_get($response->json(), 'id');
            $Boleto->update(['payment_id' => $paymentId]);

            // Corrigido aqui:
            return response()->json([
                'id' => $paymentId ?? null,
                'pedido_id' => $pedido->id ?? null,
                'boleto_id' => $Boleto->id ?? null,
                'status' => $response->status(),
                'body' => $response->json(),
            ]);
        } else {
            $idempotencyKey = Str::uuid()->toString();
            $body = [
                "transaction_amount" =>  (float) moeda2($Boleto->valor),
                "description" => count($cart) . ' ' . (count($cart) > 1 ? 'ingressos' : 'ingresso') . ' ' . $Evento->titulo,
                "external_reference" => $Boleto->id,
                "notification_url" => 'https://checkout.ucesub.com.br/retorno-mp',
                "payment_method_id" => "pix",
                "date_of_expiration" => $date,
                "payer" => [
                    "first_name" => Auth::user()->name,
                    "last_name" => Auth::user()->sobrenome,
                    "email" => Auth::user()->email,
                    "identification" => [
                        "type" => "CPF",
                        "number" => preg_replace('/\D/', '', Auth::user()->cpf)
                    ]
                ]
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $getway['access_token'],
                'Content-Type' => 'application/json',
                'X-Idempotency-Key' => $idempotencyKey,
            ])->post('https://api.mercadopago.com/v1/payments', $body);

            // dd($response);
            $dados      = $response->json();
            $paymentId  = (string) data_get($dados, 'id');
            $qrCode     = data_get($dados, 'point_of_interaction.transaction_data.qr_code');
            $qrBase64   = data_get($dados, 'point_of_interaction.transaction_data.qr_code_base64');
            $ticketUrl  = data_get($dados, 'point_of_interaction.transaction_data.ticket_url');

            if (blank($paymentId)) {
                Log::error('MP: resposta sem ID', ['dados' => $dados]);
                return back()->with('error', 'Resposta do MP sem ID do pagamento.');
            }

            $Boleto->payment_id  = $paymentId;
            $Boleto->qr_code     = $qrCode;     // string “copia e cola”
            $Boleto->cod_pix     = $qrBase64;   // imagem em base64
            $Boleto->link_ticket = $ticketUrl;

            if (!$Boleto->save()) {
                Log::error('Falha ao salvar Boleto', ['boleto_id' => $Boleto->id, 'payment_id' => $paymentId]);
                return back()->with('error', 'Não foi possível salvar dados do PIX.');
            }

            // Corrigido aqui:
            session()->forget(['cupom']);

            return response()->json([
                'id'        => $paymentId ?? null,
                'pedido_id' => $pedido->id ?? null,
                'boleto_id' => $Boleto->id ?? null,
                'status'    => $response->status(),
                'body'      => $response->json(),
            ]);
        }
    }

    public function ProcessarPagamento(Request $request)
    {
        $cupom = session()->get('cupom');
        $cart           = [];
        $totalPedido    = 0;
        // Itera sobre os dados do formulário
        foreach ($request->all() as $key => $value) {
            // Identifica os índices relacionados a cada formulário
            if (preg_match('/form-(\d+)-nome_completo/', $key, $matches)) {
                $index = $matches[1]; // Captura o índice (0, 1, 2...)

                // Coleta todos os dados para cada participante no formulário
                $nomesCompletos = $request->input("form-{$index}-nome_completo");
                $emails = $request->input("form-{$index}-email");
                $ingressosIds = $request->input("form-{$index}-ingresso");

                // Itera sobre os participantes para criar os itens do carrinho
                foreach ($nomesCompletos as $i => $nomeCompleto) {
                    $nomeCompleto = $nomeCompleto ?: Auth::user()->name . ' ' . Auth::user()->sobrenome;  // Usa nome do usuário autenticado se for nulo
                    $email = $emails[$i] ?? Auth::user()->email;         // Usa e-mail do usuário autenticado se for nulo
                    $ingressoId = $ingressosIds[$i] ?? null;

                    if ($ingressoId) {
                        // Recupera o preço do ingresso baseado no ingresso_id
                        $eventoPacote = EventoPacote::find($ingressoId);
                        $precoPacote = $eventoPacote ? $eventoPacote->preco : 0;

                        // Adiciona o participante ao carrinho
                        $cart[] = [
                            'nome_completo' => $nomeCompleto,
                            'email'         => $email,
                            'ingresso_id'   => $ingressoId,
                            'preco'         => $precoPacote
                        ];

                        // Atualiza o total do pedido
                        if ($cupom) {
                            $desconto       = $cupom['desconto'] ?? 0;
                            $precoPacote    = $precoPacote - ($precoPacote * ($desconto / 100));
                        }
                        $totalPedido += $precoPacote;
                    }
                }
            }
        }

        // dd($totalPedido);

        $taxa = [
            '1' => 2.99,
            '2' => 3.49,
            '3' => 3.49,
            '4' => 3.49,
            '5' => 3.49,
            '6' => 3.99,
            '7' => 3.99,
            '8' => 3.99,
            '9' => 3.99,
            '10' => 3.99,
        ];

        $juros = [
            '1' => 1.15,
            '2' => 1.6 * 2,
            '3' => 1.6 * 3,
            '4' => 1.6 * 4,
            '5' => 1.6 * 5,
            '6' => 1.6 * 6,
            '7' => 1.6 * 7,
            '8' => 1.6 * 8,
            '9' => 1.6 * 9,
            '10' => 1.6 * 10,
        ];

        // dd($request->all());
        $Evento = Evento::find($request->evento_id);

        $SetJuros   = $request->payment_method == 'PIX' ? 0 : $juros[$request->parcelamento];
        $SetTaxa    = $request->payment_method == 'PIX' ? 0 : $taxa[$request->parcelamento];
        $SetValorParcela = $request->payment_method == 'PIX' ? ceil((porcentagem(($SetJuros + $SetTaxa + $Evento->taxa_servico), $totalPedido) + $totalPedido)) : ceil((porcentagem(($SetJuros + $SetTaxa + $Evento->taxa_servico), $totalPedido) + $totalPedido) / $request->parcelamento);

        // Organiza os dados finais para enviar para a view ou para processamento
        $dadosCarrinho = [
            'carrinho'              => $cart,
            'total_ingressos'       => $totalPedido,
            'taxa_servico'          => $Evento->taxa_servico,
            'valor_servico'         => porcentagem($Evento->taxa_servico, $totalPedido),
            'juros'                 => $SetJuros,
            'taxa'                  => $SetTaxa,
            'total_taxas'           => $SetJuros + $SetTaxa + $Evento->taxa_servico,
            'valor_taxas'           => porcentagem(($SetJuros + $SetTaxa + $Evento->taxa_servico), $totalPedido),
            'total'                 => porcentagem(($SetJuros + $SetTaxa + $Evento->taxa_servico), $totalPedido) + $totalPedido,
            'valor_parcela'         => $SetValorParcela,
        ];


        $nomeCompleto = $request->nome_completo;
        $nomeSeparado = explode(' ', $nomeCompleto);
        $primeiroNome = $nomeSeparado[0];
        $sobreNome = implode(' ', array_slice($nomeSeparado, 1));

        $user = User::find(Auth::user()->id);
        $user->update([
            'name'          => $primeiroNome,
            'sobrenome'     => $sobreNome,
            'email'         => $request->email,
            'cpf'           => $request->cpf_cnpj,
        ]);


        if ($request->payment_method == 'PIX') {
            $pedido = Pedido::create([
                'evento_id'         => $request->evento_id,
                'user_id'           => $user->id,
                'pdv'               => 'Site',
                'qtd_pacotes'       => count($cart),
                'total'             => $dadosCarrinho['total'],
                'sub_total'         => $totalPedido,
                'taxa'              => $dadosCarrinho['valor_servico'],
                'forma_pagamento'   => $request->payment_method,
                'status'            => 'Pendente',
                'registro'          => gerarlETRARegistro(12),
            ]);
            $boleto = Boleto::create([
                'pedido_id'   => $pedido->id,
                'user_id'     => $user->id,
                'status'      => '0',
                'valor'       => $dadosCarrinho['total'],
                'ref'         => count($cart) . ' ' . (count($cart) > 1 ? 'ingressos' : 'ingresso') . ' ' . $Evento->titulo,
                'vencimento'  => now()->addDays(2),
                'taxa_lopak'  => $dadosCarrinho['valor_servico'],
            ]);
        } else {
            $pedido = Pedido::create([
                'evento_id'         => $request->evento_id,
                'user_id'           => $user->id,
                'taxa_origen'       => $Evento->taxa_servico == 0 ? 'Empresa' : 'Cliente',
                'pdv'               => 'Site',
                'qtd_pacotes'       => count($cart),
                'total'             => $dadosCarrinho['total'],
                'taxa'              => $dadosCarrinho['valor_servico'],
                'forma_pagamento'   => $request->payment_method,
                'parcelamento'      => $request->parcelamento,
                'valor_parcela'     => $dadosCarrinho['valor_parcela'],
                'status'            => 'Pendente',
                'registro'          => gerarlETRARegistro(12),
            ]);
            $boleto = Boleto::create([
                'pedido_id'   => $pedido->id,
                'user_id'     => $user->id,
                'status'      => '0',
                'valor'       => $dadosCarrinho['total'],
                'ref'         => count($cart) . ' ' . (count($cart) > 1 ? 'ingressos' : 'ingresso') . ' ' . $Evento->titulo,
                'vencimento'  => now()->addDays(2),
                'taxa_lopak'  => $dadosCarrinho['valor_servico'],
            ]);
        }

        foreach ($cart as $item) {
            $dados = [
                'pedido_id'         => $pedido->id,
                'user_id'           => $user->id,
                'evento_id'         => $request->evento_id,
                'pacote_id'         => $item['ingresso_id'],
                'status'            => '0',
                'check_in_status'   => '0',
                'check_tentativas'  => '0',
                'code_barras'       => gerarCodigo(12),
                'registro'          => gerarlETRARegistro(12),
                'num_acessos'       => 0,
                'nome_completo'     => $item['nome_completo'],
                'email'             => $item['email'],
                'preco'             => $item['preco'],
                'taxa'              => porcentagem($Evento->taxa_servico, $item['preco']),
            ];
            Ingresso::create($dados);
        }
        session()->forget(['cupom']);
        return redirect()->route('pay', [$pedido, $boleto]);
    }

    public function Pay(Pedido $Pedido, Boleto $Boleto)
    {

        return view('vendas.pay', compact('Pedido', 'Boleto'));
    }

    public function success(Pedido $Pedido)
    {
        return view('site.success', compact('Pedido'));
    }


    public function ConsultaMercadoPago($payment_id, Request $request)
    {
        $Boleto         = Boleto::where('pedido_id', $request->input('pedido_id'))->first();
        $getway         = GetWay::find($Boleto->getway_id);
        // Configura o access token
        $getway         = session()->get('getway');
        $accessToken    = $getway->access_token;

        if ($request->input('pedido_id')) {
            $pedidoId       = $request->input('pedido_id');
            $Boleto         = Boleto::where('pedido_id', $pedidoId)->first();
            $Boleto->update(['payment_id' => $payment_id]);
        }

        try {
            $response = Http::withToken($accessToken)->get("https://api.mercadopago.com/v1/payments/{$payment_id}");
            if ($response->failed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Falha ao consultar pagamento.',
                    'status' => $response->status(),
                    'body' => $response->json(),
                ], $response->status());
            }
            $dados = $response->json();
            $Boleto = Boleto::where('payment_id', $payment_id)->first();
            $Pedido = Pedido::find($Boleto->pedido_id);

            if ($dados['status'] == 'approved') {
                $Boleto->update([
                    'status'    => '1',
                    'data_pago' => date('Y-m-d'),
                ]);
                $Pedido->update(['status' => 'Aprovado']);
                $count = count($Pedido->Evento->Ingressos);
                foreach ($Pedido->Ingressos as $ingresso) {
                    $count++;

                    $ingresso->status = '1';
                    if (!empty($ingresso->Evento->rule_off)) {
                        if ($getway['tipo'] == 'pf') {
                            $ingresso->check_in_situacao = '1';
                            $Pedido->update(['check_in_situacao' => '1']);

                            if (count($Pedido->Ingressos) > 1) {
                                foreach ($Pedido->Ingressos as $ddx) {
                                    $ddx->update(['check_in_situacao' => '1']);
                                }
                            }
                        }
                    }
                    $ingresso->save();

                    // Envia e-mail
                    $user = User::find($ingresso->user_id);
                    Mail::send('emails.usuario-pdv', [
                        'name'      => $user->name,
                        'email'     => $user->email,
                        'evento'    => $Pedido->Evento->titulo,
                        'banner'    => $Pedido->Evento->banner,
                        'link'      => 'https://checkout.ucesub.com.br/print/ingressos/' . $Pedido->registro,
                        'qtd'       => count($Pedido->Ingressos),
                        'ref'       => count($Pedido->Ingressos) . ' ingresso(s) ' . $Pedido->Evento->titulo,
                    ], function ($mail) use ($user) {
                        $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                        $mail->to($user->email)->subject('Ingresso ' . env('APP_NAME'));
                    });
                }


                foreach ($user->pushToken as $push) {
                    if ($push->token) {
                        $this->sendPushNotification(
                            $push->token,
                            'Ingresso Disponível',
                            'Seu ingresso para ' . $Pedido->Evento->titulo . ' já esta disponível.',
                            [
                                'evento_id'  => 1,
                            ]
                        );
                    }
                }
            }

            return response()->json([
                'count' => $count,
                'success' => true,
                'response' => $dados['status'],
                'dados' => $dados,
                'getway' => $getway
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao consultar pagamento.',
                'error' => $e->getMessage(),
                'getway' => $getway
            ], 500);
        }
    }

    public function RetornoMercadoPago($id)
    {
        $Boleto = Boleto::with(['Pedido', 'Pedido.Ingressos', 'Pedido.Evento'])->find($id);
        if (!$Boleto) {
            return response()->json([
                'success' => false,
                'message' => 'Boleto não encontrado para o ID informado.',
            ], 404);
        }

        $pedidoId = $Boleto->pedido_id;
        $getway   = GetWay::find($Boleto->getway_id);

        if (!$getway || empty($getway->access_token)) {
            return response()->json([
                'success' => false,
                'message' => 'GetWay não encontrado ou access_token ausente.',
            ], 500);
        }

        // IDs alternativos
        $primaryGetwayId = $getway->id;
        $alternateGetwayId = $primaryGetwayId == 1 ? 2 : 1;

        // Função anônima para tentar consultar pagamento
        $consultarPagamento = function ($accessToken, $boletoId) {
            return Http::withToken($accessToken)
                ->timeout(20)
                ->get('https://api.mercadopago.com/v1/payments/search', [
                    'external_reference' => $boletoId,
                    'sort'               => 'date_created',
                    'criteria'           => 'desc',
                    'limit'              => 1,
                ]);
        };

        try {
            // 🔹 Primeira tentativa com o gateway principal
            $response = $consultarPagamento($getway->access_token, $Boleto->id);
            $payload = $response->json();

            // Se falhou ou retornou not_found, tenta o outro gateway
            if (
                $response->failed() || ($payload['status'] ?? null) == 404 ||
                (($payload['body']['error'] ?? null) === 'not_found')
            ) {

                $fallbackGetway = GetWay::find($alternateGetwayId);
                if ($fallbackGetway && !empty($fallbackGetway->access_token)) {
                    $response = $consultarPagamento($fallbackGetway->access_token, $Boleto->id);
                    $getway = $fallbackGetway; // Atualiza para o gateway alternativo
                    $payload = $response->json();
                }
            }

            if ($response->failed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Falha ao consultar pagamento.',
                    'status'  => $response->status(),
                    'body'    => $response->json(),
                ], $response->status());
            }

            $results = $payload['results'] ?? [];
            if (empty($results)) {
                return response()->json([
                    'success'  => true,
                    'response' => 'not_found',
                    'message'  => 'Nenhum pagamento encontrado para o external_reference.',
                ], 200);
            }

            $dados = $results[0];
            $paymentId = (string) data_get($dados, 'id');
            $status    = data_get($dados, 'status');

            $Boleto->forceFill(['payment_id' => $paymentId])->save();
            $Pedido = $Boleto->Pedido;

            if (!$Pedido) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pedido não encontrado.',
                ], 404);
            }

            if ($status === 'approved') {
                $Boleto->update(['status' => '1', 'data_pago' => now()->toDateString()]);
                $Pedido->update(['status' => 'Aprovado']);

                foreach ($Pedido->Ingressos as $ingresso) {
                    $ingresso->update(['status' => '1']);
                    if (!empty($ingresso->Evento->rule_off) && ($getway->tipo ?? null) === 'pf') {
                        $ingresso->update(['check_in_situacao' => '1']);
                        $Pedido->update(['check_in_situacao' => '1']);
                    }
                }

                return response()->json([
                    'success'  => true,
                    'response' => 'approved',
                    'dados'    => $dados,
                    'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
                    'redirect' => 'https://checkout.ucesub.com.br/success/' . $Pedido->id,
                ]);
            }

            return response()->json([
                'success'  => true,
                'response' => $status ?: 'unknown',
                'dados'    => $dados,
                'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao consultar pagamento.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }




    public function ValidarPagamento()
    {
        // Configura o access token
        $all = Pedido::where('status', 'Aprovado')->where('send_email', '0')->where('evento_id', 73)->get();
        $pedido = Pedido::where('status', 'Aprovado')->where('send_email', '0')->where('evento_id', 73)->first();
        if ($pedido->check_in_situacao == '1') {
            $getway = GetWay::find(2);
            $accessToken    = $getway->access_token;
        } else {
            $getway = GetWay::find(1);
            $accessToken    = $getway->access_token;
        }

        if (empty($pedido->Boleto->payment_id)) {
            $pedido->update([
                'send_email' => '1'
            ]);
            return 'Sem Boleto ' . count($all);
        }

        try {
            $response = Http::withToken($accessToken)->get("https://api.mercadopago.com/v1/payments/{$pedido->Boleto->payment_id}");
            if ($response->failed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Falha ao consultar pagamento.',
                    'status' => $response->status(),
                    'body' => $response->json(),
                ], $response->status());
            }
            $dados = $response->json();
            $Boleto = Boleto::where('payment_id', $pedido->Boleto->payment_id)->first();
            $Pedido = Pedido::find($Boleto->pedido_id);

            if ($dados['status'] == 'approved') {
                $Boleto->update([
                    'status'    => '1',
                    'data_pago' => date('Y-m-d'),
                ]);
                $pedido->update([
                    'status' => 'Aprovado',
                    'send_email' => '1'
                ]);
                $count = count($Pedido->Evento->Ingressos);
                foreach ($Pedido->Ingressos as $ingresso) {
                    $count++;

                    $ingresso->status = '1';
                    if (!empty($ingresso->Evento->rule_off)) {
                        if ($getway['tipo'] == 'pf') {
                            $ingresso->check_in_situacao = '1';
                            $Pedido->update(['check_in_situacao' => '1']);

                            if (count($Pedido->Ingressos) > 1) {
                                foreach ($Pedido->Ingressos as $ddx) {
                                    $ddx->update(['check_in_situacao' => '1']);
                                }
                            }
                        }
                    }
                    $ingresso->save();

                    // Envia e-mail
                    $user = User::find($ingresso->user_id);
                    Mail::send('emails.usuario-pdv', [
                        'name'      => $user->name,
                        'email'     => $user->email,
                        'evento'    => $Pedido->Evento->titulo,
                        'banner'    => $Pedido->Evento->banner,
                        'link'      => 'https://checkout.ucesub.com.br/print/ingressos/' . $Pedido->registro,
                        'qtd'       => count($Pedido->Ingressos),
                        'ref'       => count($Pedido->Ingressos) . ' ingresso(s) ' . $Pedido->Evento->titulo,
                    ], function ($mail) use ($user) {
                        $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                        $mail->to($user->email)->subject('Ingresso ' . env('APP_NAME'));
                    });
                }


                foreach ($user->pushToken as $push) {
                    if ($push->token) {
                        $this->sendPushNotification(
                            $push->token,
                            'Ingresso Disponível',
                            'Seu ingresso para ' . $Pedido->Evento->titulo . ' já esta disponível.',
                            [
                                'evento_id'  => 1,
                            ]
                        );
                    }
                }
            }

            return response()->json([
                'count' => count($all),
                'success' => true,
                'response' => $dados['status'],
                'dados' => $dados,
                'getway' => $getway
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao consultar pagamento.',
                'error' => $e->getMessage(),
                'getway' => $getway
            ], 500);
        }
    }

    // public function getPedido(){
    //     $pedido = Pedido::where('')
    // }

    public function aprovar($payment_id)
    {
        $boleto = Boleto::where('payment_id', $payment_id)->first();
        $boleto->update(['status' => '1']);

        $pedido     = Pedido::find($boleto->pedido_id);
        $ingressos   = Ingresso::where('pedido_id', $boleto->pedido_id)->get();
        foreach ($ingressos as $ingresso) {
            $ingresso->update(['status' => '1']);
        }
    }
}
