<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Evento;
use App\Models\Categoria;
use App\Models\Estado;
use App\Models\EventoColaborador;
use App\Models\EventoCupomDesconto;
use App\Models\EventoEventoColaboradorPacote;
use App\Models\EventoGrupoPacote;
use App\Models\EventoPacote;
use App\Models\EventoPDV;
use App\Models\Ingresso;
use App\Models\Locais;
use App\Models\Pedido;
use App\Models\Repasse;
use App\Models\RuleApp;
use App\Models\TokenPush;
use App\Models\User;
use Carbon\Carbon;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Mail;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Milon\Barcode\DNS1D;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\BorderoEventoExport;

class EventoController extends Controller
{

    protected function sendPushNotification($expoPushToken, $title, $body, $data = [])
    {
        $pushController = app(PushController::class);
        $request = new Request([
            'expoPushToken' => $expoPushToken,
            'title' => $title,
            'body' => $body,
            'data' => $data,
        ]);

        $pushController->sendNotification($request);
    }


    public function index(Evento $Evento)
    {
        if (Auth::user()->nivel == '1') {
            $pedidosPorDia = DB::table('pedidos')
                ->select('data_pedido as data', DB::raw('SUM(sub_total) as total_vendas'))
                ->where('evento_id', $Evento->id)
                ->where('status', 'Aprovado')
                ->where('check_in_situacao', '0')
                ->groupBy('data_pedido')
                ->orderBy('data_pedido')
                ->get();
        } else {
            $pedidosPorDia = DB::table('pedidos')
                ->select('data_pedido as data', DB::raw('SUM(sub_total) as total_vendas'))
                ->where('pdv_id', Auth::user()->produtor_id)
                ->where('evento_id', $Evento->id)
                ->where('status', 'Aprovado')
                ->where('check_in_situacao', '0')
                ->groupBy('data_pedido')
                ->orderBy('data_pedido')
                ->get();
        }


        // Formatar os dados para o gráfico
        $vendasDiarias = $pedidosPorDia->map(function ($registro) {
            return [
                'data' => date('d/m', strtotime($registro->data)), // Ou apenas $registro->data se já estiver como deseja
                'total' => (float) $registro->total_vendas / 100,
            ];
        });

        return view('admin.evento.index', [
            'Evento' => $Evento,
            'vendasDiarias' => $vendasDiarias,
        ]);
    }

    public function create()
    {
        $categorias = Categoria::get();
        $locais     = Locais::get();
        return view('admin.evento.create', compact('categorias', 'locais'));
    }

    public function GetLocal(Request $request)
    {
        $local = Locais::find($request->id);

        if (!$local) {
            return response()->json(['error' => 'Local não encontrado.'], 404);
        }

        return response()->json([
            'local'       => $local->local,
            'cep'         => $local->cep,
            'endereco'    => $local->endereco,
            'numero'      => $local->numero,
            'complemento' => $local->complemento,
            'bairro'      => $local->bairro,
            'cidade'      => $local->cidade,
            'estado'      => $local->estado,
            'latitude'    => $local->latitude,
            'longitude'   => $local->longitude,
        ]);
    }



    public function integration(Evento $Evento)
    {
        return view('admin.evento.integracoes', ['Evento' => $Evento]);
    }

    public function integrationUpdate(Request $request)
    {
        $evento = Evento::find($request->id);
        $evento->update($request->all());
    }


    public function store(Request $request)
    {
        $dados                  = $request->all();

        $inicio     = Carbon::createFromFormat('Y-m-d H:i', $request->data_inicio . ' ' . $request->hora_inicio);
        $fim        = Carbon::createFromFormat('Y-m-d H:i', $request->data_fim . ' ' . $request->hora_fim);

        $dados = [
            'tipo'                => $request->input('tipo') ?? 'Público',
            'titulo'              => $request->input('titulo'),
            'descricao'           => $request->input('descricao'),
            'status'              => Auth::user()->id == 1 ? 'Ativo' : 'Pausado',
            'status_musa'         => Auth::user()->id == 1 ? 'Ativo' : 'Bloqueado',
            'data_inicio'         => $inicio,
            'data_fim'            => $fim,
            'url'                 => setUri($request->input('titulo')),
            'taxa_servico'        => env('TAXA'),
        ];

        if ($request->image) {
            $nameFile = Str::of($request->titulo)->slug('-') . '.' . $request->image->getClientOriginalExtension();
            $path = $request->image->storeAs('eventos', $nameFile, 'public');
            $dados['banner'] = Storage::url($path);
        } else {
            $dados['banner'] = Storage::url('eventos/evento-sem-foto.jpg');
        }

        // Criação do evento
        $evento = Evento::create($dados);
        return redirect('/produtor/evento/pre-editar/' . $evento->id . '#ingressos');
    }

    public function show(Evento $Evento)
    {
        $pedidosPorDia = DB::table('pedidos')
            ->selectRaw('DATE(created_at) as data, SUM(sub_total) as total_vendas')
            ->where('evento_id', $Evento->id)
            ->where('status', 'Aprovado')
            ->where('check_in_situacao', '0')
            ->groupBy(DB::raw('DATE(created_at)'))
            ->orderBy('data')
            ->get();

        // Formatar os dados para o gráfico
        $vendasDiarias = $pedidosPorDia->map(function ($registro) {
            return [
                'data' => Carbon::parse($registro->data)->format('d/m'),
                'total' => (float) $registro->total_vendas / 100,
            ];
        });

        // dd($vendasDiarias);

        return view('admin.evento.index', [
            'Evento' => $Evento,
            'vendasDiarias' => $vendasDiarias,
        ]);
    }


    public function pacotes(Evento $Evento)
    {
        return view('admin.eventos.pacotes', ['Evento' => $Evento]);
    }


    public function PacoteStore(Request $request)
    {
        $dados = $request->all();
        // Verificar se data e hora já estão no formato esperado
        if (DateTime::createFromFormat('Y-m-d H:i', $dados['data_inicio'] . ' ' . $dados['hora_inicio']) === false) {
            // Concatenar data e hora para data_inicio
            $dataInicio = DateTime::createFromFormat('d/m/Y H:i', $dados['data_inicio'] . ' ' . $dados['hora_inicio']);
            if ($dataInicio) {
                $dados['data_inicio'] = $dataInicio->format('Y-m-d H:i');
            } else {
                return response()->json(['error' => 'Formato inválido para data_inicio'], 400);
            }

            // Concatenar data e hora para data_fim
            $dataFim = DateTime::createFromFormat('d/m/Y H:i', $dados['data_fim'] . ' ' . $dados['hora_fim']);
            if ($dataFim) {
                $dados['data_fim'] = $dataFim->format('Y-m-d H:i');
            } else {
                return response()->json(['error' => 'Formato inválido para data_fim'], 400);
            }
        } else {
            // Data e hora já estão no formato correto
            $dados['data_inicio'] = $dados['data_inicio'] . ' ' . $dados['hora_inicio'];
            $dados['data_fim'] = $dados['data_fim'] . ' ' . $dados['hora_fim'];
        }

        // Remover formatação de moeda do preço
        $dados['preco'] = tiraMoeda($request->preco);
        $dados['status'] = 'Ativo';

        // Criar o pacote principal
        $pacote = EventoPacote::create($dados);

        // Verificar se existem informações de meia-entrada
        if (isset($dados['nome_meia_pago']) && isset($dados['qtd_meia_pago']) && isset($dados['valor_meia_pago'])) {
            // Criar o pacote de meia-entrada
            $dadosMeiaEntrada = $dados;
            $dadosMeiaEntrada['nome'] = $dados['nome_meia_pago'];
            $dadosMeiaEntrada['qtd'] = $dados['qtd_meia_pago'];
            $dadosMeiaEntrada['preco'] = tiraMoeda($dados['valor_meia_pago']); // Remover a formatação da moeda

            // Criar o pacote de meia-entrada
            EventoPacote::create($dadosMeiaEntrada);
        }

        // Encontrar o evento e retornar a view
        $Evento = Evento::find($request->evento_id);
        return view('admin.evento.ajax.pacotes', ['Evento' => $Evento]);
    }


    public function IngressoUpdate(Request $request)
    {
        $dados = $request->all();
        $pacote = EventoPacote::find($request->pacote_id);

        if ($request->tipo == 'Excluir') {
            if (count($pacote->Ingressos) > 0) {
                return 'error_excluir';
            } else {
                $pacote->delete();
            }
        } else {
            // Verificar se data e hora já estão no formato esperado
            if (DateTime::createFromFormat('Y-m-d H:i', $dados['data_inicio'] . ' ' . $dados['hora_inicio']) === false) {
                // Concatenar data e hora para data_inicio
                $dataInicio = DateTime::createFromFormat('d/m/Y H:i', $dados['data_inicio'] . ' ' . $dados['hora_inicio']);
                if ($dataInicio) {
                    $dados['data_inicio'] = $dataInicio->format('Y-m-d H:i');
                } else {
                    return response()->json(['error' => 'Formato inválido para data_inicio'], 400);
                }

                // Concatenar data e hora para data_fim
                $dataFim = DateTime::createFromFormat('d/m/Y H:i', $dados['data_fim'] . ' ' . $dados['hora_fim']);
                if ($dataFim) {
                    $dados['data_fim'] = $dataFim->format('Y-m-d H:i');
                } else {
                    return response()->json(['error' => 'Formato inválido para data_fim'], 400);
                }
            } else {
                // Data e hora já estão no formato correto
                $dados['data_inicio'] = $dados['data_inicio'] . ' ' . $dados['hora_inicio'];
                $dados['data_fim'] = $dados['data_fim'] . ' ' . $dados['hora_fim'];
            }

            // Remover formatação de moeda do preço
            $dados['preco'] = tiraMoeda($request->preco);
            $dados['status'] = 'Ativo';

            // dd($request->all(),$dados);
            $pacote->update($dados);
        }

        // Encontrar o evento e retornar a view
        $Evento = Evento::find($request->evento_id);
        return view('admin.evento.ajax.pacotes', ['Evento' => $Evento]);
    }

    public function PacoteShow(Request $request)
    {
        $exe = EventoPacote::find($request->id);

        // dd($exe);
        return view('admin.evento.ajax.pacote-show', ['exe' => $exe]);
    }

    public function PacoteDestroy(Request $request)
    {
        $exe = EventoPacote::find($request->id);
        $exe->delete();
    }

    public function PacoteUpdateStatus(Request $request)
    {
        $exe = EventoPacote::find($request->id);
        $exe->update([
            'status' => $exe->status == 'Ativo' ? 'Inativo' : 'Ativo'
        ]);

        $Evento = Evento::find($exe->evento_id);
        return view('admin.evento.ajax.pacotes', ['Evento' => $Evento]);
    }

    public function GrupoStore(Request $request)
    {
        EventoGrupoPacote::create($request->all());
        $evento = Evento::find($request->evento_id);
        return view('admin.eventos.ajax.grupos', ['evento' => $evento]);
    }


    public function edit(Evento $Evento)
    {
        return view('admin.evento.editar', ['Evento' => $Evento]);
    }

    public function update(Request $request, Evento $Evento)
    {
        $dados = $request->all();
        if ($request->atualziar_mensagem_ingresso) {
            $Evento->update(['mensagem_ingresso' => $request->mensagem_ingresso]);
            return redirect()->back()->with('success', 'Atualizado com sucesso');
        }

        // Converter data_inicio
        $dados['data_inicio']   = DateTime::createFromFormat('Y-m-d H:i', $request->data_inicio . ' ' . $request->hora_inicio);

        // Converter data_fim
        $dados['data_fim']      = DateTime::createFromFormat('Y-m-d H:i', $request->data_fim . ' ' . $request->hora_fim);

        if ($request->image) {
            $nameFile = Str::of($request->titulo)->slug('-') . rand() . '.' . $request->image->getClientOriginalExtension();
            $path = $request->image->storeAs('eventos', $nameFile, 'public');
            $dados['banner'] = Storage::url($path);
        } else {
            $dados['banner'] = Storage::url('eventos/evento-sem-foto.jpg');
        }

        // Upload do mapa do evento
        if ($request->mapa) {
            $nameFileMapa = Str::of($request->titulo)->slug('-') . '-mapa.' . $request->mapa->getClientOriginalExtension();
            $pathMapa = $request->mapa->storeAs('eventos', $nameFileMapa, 'public');
            $dados['mapa_evento'] = Storage::url($pathMapa);
        }


        // dd($request->all(),$dados);

        $Evento->update($dados);
        return redirect()->back()->with('success', 'Atualizado com sucesso');
    }

    public function destroy(Request $request)
    {
        $exe = Evento::find($request->id);
        // dd(count($exe->Ingressos));
        if (count($exe->Ingressos) > 0) {
            $exe->update(['status' => 'Excluído']);
        } else {
            $exe->delete();
        }
    }

    public function colaboradores(Evento $Evento)
    {
        return view('admin.evento.colaboradores', ['Evento' => $Evento]);
    }

    public function pdvs(Evento $Evento)
    {
        return view('admin.evento.pdvs', ['Evento' => $Evento]);
    }

    public function ingressos(Evento $Evento)
    {
        return view('admin.evento.ingressos', ['Evento' => $Evento]);
    }

    public function MSGIngressos(Evento $Evento)
    {
        return view('admin.evento.mensagem-do-ingresso', ['Evento' => $Evento]);
    }

    public function participantes(Request $request, Evento $Evento)
    {
        $limit = 50;
        $offset = $request->input('offset', 0);

        if (Auth::user()->nivel == '1') {
            $ingressos = Ingresso::where('status', '1')
                ->where('evento_id', $Evento->id)
                ->offset($offset)
                ->limit($limit)
                ->get();
        } else {
            $ingressos = Ingresso::where('status', '1')
                ->where('evento_id', $Evento->id)
                ->where('pdv_id', Auth::user()->produtor_id)
                ->offset($offset)
                ->limit($limit)
                ->get();
        }

        if ($request->ajax()) {
            $html = view('admin.evento.ajax.participantes', compact('ingressos'))->render();
            return response()->json(['html' => $html, 'count' => $ingressos->count()]);
        }

        return view('admin.evento.participantes', [
            'Evento' => $Evento,
            'ingressos' => $ingressos,
        ]);
    }


    public function checkin(Evento $Evento)
    {
        $totalIngressosDisponiveis      = $Evento->Pacotes->sum('qtd');
        $CheckinRealizados         = count($Evento->Ingressos->where('check_in_status', '1'));
        $Realizados                       = MediaVendas($CheckinRealizados, $totalIngressosDisponiveis); // 2.86
        $NaoRealizados                    = 100 - MediaVendas($CheckinRealizados, $totalIngressosDisponiveis); // 2.86

        return view('admin.evento.checkin', ['Evento' => $Evento, 'Realizados' => $Realizados, 'NaoRealizados' => $NaoRealizados]);
    }

    public function checkinOnline(Evento $Evento)
    {
        return view('admin.evento.checkin-online', ['Evento' => $Evento]);
    }

    public function pdv(Evento $Evento)
    {
        return view('admin.evento.pdv', ['Evento' => $Evento]);
    }

    public function ColaboradorStore(Request $request)
    {
        $permissoesMap = [
            '1'   => 'EditarEvento',
            '2'   => 'CheCkin',
            '3'   => 'Ingressos',
            '4'   => 'Participantes',
            '5'   => 'Cortesias',
            '6'   => 'Colaboradores',
            '7'   => 'Cupons',
            '8'   => 'PDV',
            '10'  => 'NotificarClientes',
            '11'  => 'Relatorios',
            '12'  => 'Financeiro',
            '13'  => 'Extrato',
            '100' => 'Todos',
        ];

        // dd($request->all());

        // Validação básica
        if (!$request->has(['email', 'id', 'pacotes.permissoes', 'pacotes.pacotes'])) {
            return response()->json(['error' => 'Dados incompletos'], 422);
        }

        $user = User::where('email', $request->email)->first();
        $Evento = Evento::find($request->id);

        if (!$user) {
            return response()->json(['error' => 'Usuário não encontrado'], 404);
        }

        if (!$Evento) {
            return response()->json(['error' => 'Evento não encontrado'], 404);
        }

        // Se marcou a permissão "Todos" e o evento não tem produtor, define o usuário como produtor
        if (
            in_array('100', $request->pacotes['permissoes']) &&
            (empty($Evento->produtor_id) || $Evento->produtor_id == 1)
        ) {
            $Evento->update(['produtor_id' => $user->id]);
        }

        // Verifica se já existe o vínculo
        $colaborador = EventoColaborador::where('evento_id', $Evento->id)
            ->where('user_id', $user->id)
            ->first();

        if (!$colaborador) {
            $colaborador = EventoColaborador::create([
                'evento_id' => $Evento->id,
                'user_id'   => $user->id,
            ]);
        } else {
            // Limpa permissões e pacotes antigos
            EventoEventoColaboradorPacote::where('evento_colaborador_id', $colaborador->id)->delete();

            RuleApp::where('evento_id', $Evento->id)
                ->where('user_id', $user->id)
                ->delete();
        }

        // Salva os pacotes
        if (!empty($request->pacotes['pacotes']) && is_array($request->pacotes['pacotes'])) {
            foreach ($request->pacotes['pacotes'] as $pacoteId) {
                EventoEventoColaboradorPacote::firstOrCreate([
                    'evento_colaborador_id' => $colaborador->id,
                    'pacote_id'             => $pacoteId,
                    'user_id'               => $user->id,
                ]);
            }
        }

        // Salva as permissões
        if (!empty($request->pacotes['permissoes']) && is_array($request->pacotes['permissoes'])) {
            foreach ($request->pacotes['permissoes'] as $permissaoId) {
                if (isset($permissoesMap[$permissaoId])) {
                    RuleApp::create([
                        'evento_id' => $Evento->id,
                        'user_id'   => $user->id,
                        'menu'      => $permissoesMap[$permissaoId],
                        'permissao' => '1',
                    ]);
                }
            }
        }

        return view('admin.evento.ajax.colaboradores', ['Evento' => $Evento]);
    }

    public function ColaboradorShow(Request $request)
    {
        $colaborador = EventoColaborador::find($request->id);
        $Evento = Evento::find($colaborador->evento_id);
        return view('admin.evento.ajax.colaborador-show', ['colaborador' => $colaborador, 'Evento' => $Evento]);
    }

    public function ColaboradorUpdate(Request $request)
    {
        // Localiza o colaborador pelo ID
        $colaborador = EventoColaborador::find($request->id);

        if (!$colaborador) {
            return response()->json(['error' => 'Colaborador não encontrado'], 404);
        }

        // Remove os registros antigos relacionados ao evento_colaborador_id
        EventoEventoColaboradorPacote::where('evento_colaborador_id', $request->id)->delete();
        if (!empty($request->pacotes)) {
            foreach ($request->pacotes as $pacote) {
                EventoEventoColaboradorPacote::create([
                    'evento_colaborador_id' => $request->id,
                    'pacote_id'             => $pacote,
                    'user_id'               => $colaborador->user_id,
                ]);
            }
        }
        $Evento = Evento::find($colaborador->evento_id);
        return view('admin.evento.ajax.colaboradores', ['Evento' => $Evento]);
    }

    public function ColaboradorDestroy(Request $request)
    {
        $colaborador = EventoColaborador::find($request->id);
        RuleApp::where('evento_id', $colaborador->evento_id)
            ->where('user_id', $colaborador->user_id)
            ->delete();
        $colaborador->delete();
    }


    public function PDVStore(Request $request)
    {
        $request->validate([
            'id' => 'required|integer', // evento_id
            'nome' => 'required|string',
            'tipos_ingressos' => 'required|array',
            'colaboradores' => 'required|array',
        ]);

        $eventoId = $request->id;
        $nome = $request->nome;
        $tiposIngressos = $request->tipos_ingressos;
        $colaboradores = $request->colaboradores;

        foreach ($colaboradores as $colaboradorId) {
            foreach ($tiposIngressos as $tipoIngressoId) {
                $eventoPDV = EventoPDV::where('user_id', $colaboradorId)
                    ->where('evento_id', $eventoId)
                    ->where('pacote_id', $tipoIngressoId)
                    ->first();

                if ($eventoPDV) {
                    $eventoPDV->update([
                        'nome' => $nome,
                        'status' => '1',
                    ]);
                } else {
                    EventoPDV::create([
                        'user_id' => $colaboradorId,
                        'evento_id' => $eventoId,
                        'pacote_id' => $tipoIngressoId,
                        'nome' => $nome,
                        'status' => '1',
                    ]);
                }
            }
        }

        $Evento = Evento::find($request->id);
        return view('admin.evento.ajax.pdvs', ['Evento' => $Evento]);
    }

    public function AddCarrinho(Request $request)
    {
        $pacote = EventoPacote::find($request->id);

        if (!$pacote) {
            return response()->json(['error' => 'Pacote não encontrado'], 404);
        }

        $cart = $request->session()->get('cart', []);

        // Se a quantidade for 0, remove o item do carrinho
        if ($request->quantidade == 0) {
            if (isset($cart[$pacote->id])) {
                unset($cart[$pacote->id]); // Remove o item do carrinho
            }
            $request->session()->put('cart', $cart);
            $this->salvarCarrinhoNoBancoDeDados($request); // Atualiza o banco de dados
        }

        // Adiciona ou atualiza o item no carrinho
        $cart[$pacote->id] = [
            'pacote_id' => $pacote->id,
            'pacote' => $pacote->nome,
            'preco' => $pacote->preco,
            'qtd' => $request->quantidade,
            'total' => $request->quantidade * $pacote->preco,
        ];

        $request->session()->put('cart', $cart);
        $this->salvarCarrinhoNoBancoDeDados($request);

        $cart       = session()->get('cart');
        $cliente    = session()->get('cliente');
        $pagamento  = session()->get('pagamento');
        $total      = 0;
        if ($cart) {
            foreach ($cart as $item) {
                $total += $item['total'];
            }
        }
        return view('admin.evento.ajax.resumo_pdv', ['total' => $total, 'cliente' => $cliente, 'pagamento' => $pagamento]);
    }


    public function AddUser(Request $request)
    {
        $cliente = $request->session()->get('cliente');
        $cliente = [
            'name' => $request->name,
            'email' => $request->email,
        ];
        $request->session()->put('cliente', $cliente);

        $cart       = session()->get('cart');
        $cliente    = session()->get('cliente');
        $pagamento  = session()->get('pagamento');
        $total      = 0;
        if ($cart) {
            foreach ($cart as $item) {
                $total += $item['total'];
            }
        }
        return view('admin.evento.ajax.resumo_pdv', ['total' => $total, 'cliente' => $cliente, 'pagamento' => $pagamento]);
    }

    public function AddPagamento(Request $request)
    {
        $pagamento = $request->session()->get('pagamento');
        $pagamento = [
            'forma_pagamento'   => $request->forma_pagamento,
            'parcelamento'      => $request->parcelamento,
        ];
        $request->session()->put('pagamento', $pagamento);

        $cart       = session()->get('cart');
        $cliente    = session()->get('cliente');
        $pagamento  = session()->get('pagamento');
        $total      = 0;
        if ($cart) {
            foreach ($cart as $item) {
                $total += $item['total'];
            }
        }
        return view('admin.evento.ajax.resumo_pdv', ['total' => $total, 'cliente' => $cliente, 'pagamento' => $pagamento]);
    }

    private function salvarCarrinhoNoBancoDeDados(Request $request)
    {
        // Salva o carrinho no banco de dados de sessões
        $sessionId = $request->session()->getId();
        $payload = base64_encode(serialize($request->session()->all()));

        DB::table('sessions')->updateOrInsert(
            ['id' => $sessionId],
            [
                'payload' => $payload,
                'last_activity' => now()->timestamp // Gera um timestamp UNIX
            ]
        );
    }

    public function PdvDestroy(Request $request)
    {
        $request->session()->forget(['cart', 'cliente', 'pagamento']);
    }

    public function ValidarSessoes(Request $request)
    {
        $cart       = session()->get('cart');
        $cliente    = session()->get('cliente');
        $pagamento  = session()->get('pagamento');

        if (empty($cart)) {
            return response()->json(['success' => false, 'message' => 'O carrinho está vazio.']);
        }

        if (empty($cliente)) {
            return response()->json(['success' => false, 'message' => 'Os dados do cliente não foram preenchidos.']);
        }

        if (empty($pagamento)) {
            return response()->json(['success' => false, 'message' => 'Os dados de pagamento não foram preenchidos.']);
        }
        return response()->json(['success' => true]);
    }

    public function ParticpanteStore(Request $request)
    {

        for ($i = 1; $i < 1000; $i++) {
            $pedido = Pedido::create([
                'evento_id'         => 29,
                'user_id'           => 5,
                'qtd_pacotes'       => 1,
                'total'             => 0,
                'taxa'              => 0,
                'status'            => 'Aprovado',
                'forma_pagamento'   => 'Manual',
                'parcelamento'      => '1',
                'registro'          => gerarlETRARegistro(12),
                'pdv_user'          => Auth::user()->id,
                'pdv'               => 'Manual',
                'taxa_origen'       => 'Empresa',
            ]);

            // Cria ingresso
            $ingresso = Ingresso::create([
                'nome_completo'     => 'Roberto Brito',
                'email'             => 'rbetobrito@gmail.com',
                'pedido_id'         => $pedido->id,
                'user_id'           => 5,
                'evento_id'         => 29,
                'pacote_id'         => 80,
                'pdv'               => 'Manual',
                'preco'             => 0,
                'taxa'              => 0,
                'valor_final'       => 0,
                'status'            => '1',
                'code_barras'       => gerarCodigo(12),
                'registro'          => gerarlETRARegistro(12),
                'check_in_status'   => '0',
                'check_tentativas'  => '0',
            ]);
        }

        return;

        $Evento = Evento::findOrFail($request->evento_id);
        $emails = $request->emails;
        $nomes = $request->nomes;
        $cpfs = $request->cpfs;
        $pacotes = $request->pacotes;

        foreach ($emails as $index => $email) {
            $user = User::where('email', $email)->first();
            $isNovoUsuario = false;
            $password = null;

            if (empty($user)) {
                // Divide nome e sobrenome
                $nomeCompleto = trim($nomes[$index]);
                $nomePartes = explode(' ', $nomeCompleto, 2);
                $primeiroNome = $nomePartes[0];
                $sobreNome = $nomePartes[1] ?? '';

                // Cria usuário
                $password = gerarCodigo(6);
                $user = User::create([
                    'nivel'     => '3',
                    'status'    => 'Nunca Logou',
                    'name'      => $primeiroNome,
                    'sobrenome' => $sobreNome,
                    'email'     => $email,
                    'password'  => bcrypt($password),
                ]);

                $isNovoUsuario = true;
            }

            // Cria Pedido
            $pedido = Pedido::create([
                'evento_id'         => $request->evento_id,
                'user_id'           => $user->id,
                'qtd_pacotes'       => 1,
                'total'             => 0,
                'taxa'              => 0,
                'status'            => 'Aprovado',
                'forma_pagamento'   => 'Manual',
                'parcelamento'      => '1',
                'registro'          => gerarlETRARegistro(12),
                'pdv_user'          => Auth::user()->id,
                'pdv'               => 'Manual',
                'taxa_origen'       => 'Empresa',
            ]);

            // Cria ingresso
            $ingresso = Ingresso::create([
                'nome_completo'     => $nomes[$index],
                'email'             => $email,
                'pedido_id'         => $pedido->id,
                'user_id'           => $user->id,
                'evento_id'         => $request->evento_id,
                'pacote_id'         => $pacotes[$index],
                'pdv'               => 'Manual',
                'preco'             => 0,
                'taxa'              => 0,
                'valor_final'       => 0,
                'status'            => '1',
                'code_barras'       => gerarCodigo(12),
                'registro'          => gerarlETRARegistro(12),
                'check_in_status'   => '0',
                'check_tentativas'  => '0',
            ]);

            // Envia e-mail com o ingresso (e senha se for novo usuário)
            Mail::send('emails.usuario-pdv', [
                'name'      => $user->name,
                'email'     => $user->email,
                'evento'    => $Evento->titulo,
                'banner'    => $Evento->banner,
                'link'      => 'https://checkout.ucesub.com.br/print/ingressos/' . $pedido->registro,
                'qtd'       => '1',
                'ref'       => $Evento->titulo
            ], function ($mail) use ($email, $Evento) {
                $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                $mail->to($email)->subject('Seu ingresso para ' . $Evento->titulo . ' chegou - ' . env('APP_NAME'));
            });
        }

        return response()->json(['status' => true, 'mensagem' => 'Ingressos e pedidos gerados com sucesso.']);
    }

    public function CheckinStore(Request $request)
    {
        $exe = Ingresso::find($request->id);
        $exe->update([
            'check_in_status'   => '1',
            'check_in_data'     => now(),
            'check_tentativas'  => $exe->check_tentativas + 1,
        ]);
        $Evento = Evento::find($exe->evento_id);
        return view('admin.evento.ajax.check-in-online', ['Evento' => $Evento]);
    }

    public function AtivarTaxa(Request $request)
    {
        $Evento = Evento::find($request->id);
        if ($Evento->taxa_servico == 10) {
            $Evento->update(['taxa_servico' => 0]);
        } else {
            $Evento->update(['taxa_servico' => 10]);
        }
        return response()->json([
            'html' => view('admin.evento.ajax.taxa-servico', ['Evento' => $Evento])->render(),
            'nova_taxa' => $Evento->taxa_servico,
        ]);
    }

    public function SendIngressoEmail(Request $request)
    {
        $ingresso = Ingresso::with(['Evento', 'Pacote'])->findOrFail($request->id);
        $user = User::find($ingresso->user_id); // Altere para o usuário correto
        Mail::send('emails.ingresso', [
            'ingresso' => $ingresso,
            'user' => $user,
        ], function ($mail) use ($user) {
            $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
            $mail->to($user->email)->subject('Carteira de Meia - ' . env('APP_NAME'));
        });
    }

    public function cupons(Evento $Evento)
    {
        return view('admin.evento.cupons', ['Evento' => $Evento]);
    }

    public function CupomStore(Request $request)
    {
        $dados = $request->all();
        $divulgadores = explode(',', strtoupper($request->cupom)); // separa e deixa em maiúsculo

        foreach ($divulgadores as $divulgador) {
            EventoCupomDesconto::create([
                'evento_id' => $request->evento_id,
                'cupom'     => trim($divulgador),
                'desconto'  => $request->desconto,
            ]);
        }

        $cupons = EventoCupomDesconto::where('evento_id', $request->evento_id)->get();
        return view('admin.evento.ajax.cupons', compact('cupons'));
    }

    public function CupomDestroy(Request $request)
    {
        $dados = $request->all();
        $Cupom = EventoCupomDesconto::find($request->id);

        if (count($Cupom->Pedidos) > 0) {
            $Cupom->update([
                'desconto'  => 0,
                'status'  => '0',
            ]);

            dd($Cupom);
        } else {
            $Cupom->delete();
        }
    }

    public function relatorio()
    {
        return 404;
    }

    public function cortesias(Evento $Evento)
    {
        return view('admin.evento.cortesias', ['Evento' => $Evento]);
    }


    public function BuscarParticipante(Request $request)
    {
        $ingressos = Ingresso::where('evento_id', $request->evento_id)
            ->where(function ($query) use ($request) {
                $query->where('nome_completo', 'LIKE', '%' . $request->s . '%')
                    ->orWhere('registro', 'LIKE', '%' . $request->s . '%')
                    ->orWhere('code_barras', 'LIKE', '%' . $request->s . '%');
            })
            ->where('status', '1') // opcional, caso queira filtrar só os ativos
            ->get(); // lembre-se de executar a query

        return view('admin.evento.ajax.participantes', compact('ingressos'));
    }

    public function EmailMarketing(Evento $Evento)
    {
        return view('admin.evento.email-marketing', ['Evento' => $Evento]);
    }

    public function repasses(Evento $Evento)
    {
        return view('admin.evento.repasses', ['Evento' => $Evento]);
    }

    public function RepasseStore(Request $request)
    {
        $dados = $request->all();
        $comprovantePath = null;
        if ($request->hasFile('comprovante_pagamento')) {
            $nameFile = Str::of('comprovante_pagamento')->slug('-') . rand() . '.' . $request->comprovante_pagamento->getClientOriginalExtension();
            $path = $request->comprovante_pagamento->storeAs('repasses', $nameFile);
            $dados['comprovante'] = env('BASE_STORAGE') . $path;
        }

        $dados['user_id']       = $request['user_id'];
        $dados['evento_id']     = $request['evento_id'];
        $dados['valor']         = tiraMoeda($request['valor']);
        $dados['observacoes']   = $request['observacoes'];
        Repasse::create($dados);

        return back()->with('success', 'Repasse registrado com sucesso!');
    }


    public function bordero(Evento $Evento)
    {
        return Excel::download(
            new BorderoEventoExport($Evento->id),
            'bordero_evento_' . $Evento->titulo . '.xlsx'
        );
    }
}
