<?php

namespace App\Http\Controllers;

use App\Models\Boleto;
use App\Models\Estado;
use App\Models\Evento;
use App\Models\EventoPacote;
use App\Models\Ingresso;
use App\Models\Pedido;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;
use Illuminate\Support\Facades\Storage;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Carbon\Carbon;
use Symfony\Contracts\EventDispatcher\Event;

class CarteiraController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $carteiras = Pedido::where('pdv_id', Auth::user()->id)->where('status', 'ATIVO')->take(1000)->orderby('id', 'desc')->get();
        return view('admin.carteiras.index', compact('carteiras'));
    }

    public function site()
    {
        $carteiras = DB::connection('second_db')
            ->table('pedidos')
            ->orderBy('id', 'desc')
            ->take(5)
            ->get();


        return view('admin.carteiras.pedidos-site', compact('carteiras'));
    }


    public function result(Request $request)
    {
        $busca = $request->input('cliente');

        // -------------------------------------------------------
        // 1. BUSCAR USUÁRIO NO BANCO PRIMÁRIO (apenas name)
        // -------------------------------------------------------
        $user = \App\Models\User::where('name', 'LIKE', "%{$busca}%")
            ->orWhere('cpf', 'LIKE', "%{$busca}%")
            ->orWhere('email', 'LIKE', "%{$busca}%")
            ->first();

        $origem = 'primary';

        // -------------------------------------------------------
        // 2. BUSCAR NO BANCO SECUNDÁRIO (name + sobrenome)
        // -------------------------------------------------------
        if (!$user) {
            $user = \App\Models\User::on('second_db')
                ->whereRaw("CONCAT(name, ' ', sobrenome) LIKE ?", ["%{$busca}%"])
                ->orWhere('cpf', 'LIKE', "%{$busca}%")
                ->orWhere('email', 'LIKE', "%{$busca}%")
                ->first();

            if ($user) {
                $origem = 'second';
            }
        }

        // -------------------------------------------------------
        // 3. USUÁRIO NÃO ENCONTRADO
        // -------------------------------------------------------
        if (!$user) {
            return view('admin.carteiras.ajax.carteiras', [
                'carteiras' => [],
                'mensagem' => 'Nenhum usuário encontrado.'
            ]);
        }

        // -------------------------------------------------------
        // 4. BUSCAR PEDIDOS
        // -------------------------------------------------------
        if ($origem === 'primary') {
            $carteiras = DB::table('pedidos_novo')
                ->where('user_id', $user->id)
                ->get();
        } else {
            $carteiras = DB::connection('second_db')
                ->table('pedidos')
                ->where('user_id', $user->id)
                ->get();
        }

        // -------------------------------------------------------
        // 5. BUSCAR ESTUDANTE MANUALMENTE PARA CADA PEDIDO
        // -------------------------------------------------------
        foreach ($carteiras as $c) {

            if ($origem === 'primary') {
                // estudante na base principal
                $c->Estudante = \App\Models\User
                    ::where('id', $c->user_id)
                    ->first();
            } else {
                // estudante na base secundária
                $c->Estudante = \App\Models\User::on('second_db')
                    ->where('id', $c->user_id)
                    ->first();
            }
        }

        // DD($carteiras);

        return view('admin.carteiras.ajax.carteiras', compact('carteiras', 'user', 'origem'));
    }




    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $estados = Estado::all();
        return view("admin.carteiras.create", compact('estados'));
    }

    public function store(Request $request)
    {

        // ===============================
        // 1. VALIDAÇÕES BÁSICAS
        // ===============================
        if (!$request->hasFile('image')) {
            return back()->withErrors(['error' => 'A foto é obrigatória']);
        }

        // ===============================
        // 2. UPLOAD DA IMAGEM
        // ===============================
        $file = $request->file('image');
        $nomeFoto = 'fotos/' . time() . '-' . uniqid() . '.jpg';
        $manager = new ImageManager(new Driver());
        $image = $manager
            ->read($file)
            ->toJpeg(90);
        Storage::disk('local')->put($nomeFoto, $image);

        // ===============================
        // 3. BUSCAR USUÁRIO PELO EMAIL
        // ===============================
        $email = $request->email ?? setUri($request->name) . setUri($request->cpf) . '@ucesub.com.br';
        $user = User::where('email', $email)->first();
        $pacote = EventoPacote::find($request->pacote_id);


        do {
            $registro = gerarLetraRegistro(6);
        } while (Pedido::where('registro', $registro)->exists());


        // ===============================
        // 4. CRIA OU ATUALIZA USUÁRIO
        // ===============================
        if ($user) {
            // ATUALIZA
            $user->update([
                "pdv_id"      => Auth::user()->produtor_id,
                "status"      => 'ATIVO',
                "name"        => $request->name,
                "cpf"         => $request->cpf,
                "celular"     => $request->celular,
                "nascimento"  => Carbon::createFromFormat('d/m/Y', $request->nascimento)->format('Y-m-d'),
                "foto"        => $nomeFoto,
            ]);
        } else {
            // CRIA
            $user = User::create([
                "pdv_id"      => Auth::user()->produtor_id,
                "status"      => 'ATIVO',
                "nivel"       => '3',
                "name"        => $request->name,
                "email"       => $email,
                "password"    => bcrypt('12345678'),
                "cpf"         => $request->cpf,
                "celular"     => $request->celular,
                "nascimento"  => Carbon::createFromFormat('d/m/Y', $request->nascimento)->format('Y-m-d'),
                "storage"     => env('BASE_STORAGE') . '/',
                "foto"        => env('BASE_STORAGE') . '/' . $nomeFoto,
                "saldo"       => 0,
                "saldo_bloqueado" => 0,
            ]);
        }


        // ===============================
        // 5. CRIAR CARTEIRA (PEDIDO)
        // ===============================
        $pedido = Pedido::create([
            'pdv_id'          => Auth::user()->produtor_id,
            'evento_id'       => $pacote->evento_id,
            'user_id'         => $user->id,
            'taxa_origen'     => 'Empresa',
            'pdv'             => 'Manual',
            'qtd_pacotes'     => 1,
            'total'           => $pacote->preco + porcentagem($pacote->Evento->taxa_servico, $pacote->preco),                       // total final em centavos
            'sub_total'       => $pacote->preco,
            'taxa'            => porcentagem($pacote->Evento->taxa_servico, $pacote->preco),
            'forma_pagamento' => $request->forma_pagamento,
            'status'          => 'Aprovado',
            'registro'        => $registro,
        ]);

        // ===============================
        // 6. CANCELAR CARTEIRAS ANTIGAS
        // ===============================
        $user->Carteiras()
            ->where('id', '!=', $pedido->id)
            ->update(['status' => 'CANCELADA']);


        // ===================== CRIA BOLETO =====================
        $boleto = Boleto::create([
            'pedido_id'   => $pedido->id,
            'user_id'     => $user->id,
            'status'      => '1',
            'valor'       => $pacote->preco + porcentagem($pacote->Evento->taxa_servico, $pacote->preco), // sempre em centavos
            'ref'         => $pacote->Evento->titulo . ' (' . $pacote->nome . ')',
            'vencimento'  => now()->addDays(2),
        ]);


        do {
            $registroIngresso = gerarLetraRegistro(6);
        } while (Ingresso::where('registro', $registroIngresso)->exists());

        do {
            $codeBarras = gerarCodigo(6);
        } while (Ingresso::where('code_barras', $codeBarras)->exists());


        $ingresso = Ingresso::create([
            'pdv_id'                => Auth::user()->produtor_id,
            'pedido_id'             => $pedido->id,
            'user_id'               => $user->id,
            'evento_id'             => $pacote->Evento->id,
            'pacote_id'             => $pacote->id,
            'pdv'                   => 'Manual',
            'status'                => '1',
            'check_in_status'       => '0',
            'check_tentativas'      => 0,
            'code_barras'           => $codeBarras,
            'registro'              => $registroIngresso,
            'num_acessos'           => 0,
            'nome_completo'         => $user->name . ' ' . $user->sobrenome,
            'email'                 => $user->email,
            'cpf'                   => $user->cpf,
            'celular'               => $user->celular,
            'preco'                 => $pacote->preco,
            'taxa'                  => porcentagem($pacote->Evento->taxa_servico, $pacote->preco),
            'valor_final'           => $pacote->preco + porcentagem($pacote->Evento->taxa_servico, $pacote->preco),
            'instituicao_ensino' => $request->instituicao_ensino,
            'curso'              => $request->curso,
            'nivel_ensino'       => $request->nivel_ensino,
            'foto'               => env('BASE_STORAGE') . '/' . $nomeFoto,
        ]);

        return redirect()->route('editar.pedido', $pedido->id)->with('success', 'Carteira cadastrada com sucesso!');
    }

    public function editar($id)
    {
        $carteira = Pedido::with(['IngressosCliente'])->findOrFail($id);

        // sempre será 1 ingresso
        $ingresso = $carteira->IngressosCliente->first();

        // dd($carteira, $ingresso);
        $Evento = Evento::find($carteira->evento_id);

        // dd($ingresso->GetOriginal());

        return view('admin.evento.editar-carteira', [
            'carteira' => $carteira,
            'ingresso' => $ingresso,
            'Evento' => $Evento,
        ]);
    }


    public function update(Request $request, $id)
    {
        // 1. BUSCA PEDIDO + INGRESSO + USER
        $pedido = Pedido::with(['IngressosCliente', 'Cliente'])->findOrFail($id);

        $nomeFoto = null;

        // 2. UPLOAD DA IMAGEM (se houver)
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $nomeFoto = 'fotos/' . time() . '-' . uniqid() . '.jpg';

            $manager = new ImageManager(new Driver());
            $image = $manager->read($file)->toJpeg(90);

            Storage::disk('local')->put($nomeFoto, $image);
        }

        // 3. ATUALIZA USUÁRIO
        $user = User::find($pedido->user_id);
        $dadosUsuario = [
            "pdv_id"     => Auth::id(),
            "status"     => 'ATIVO',
            "name"       => $request->name,
            "cpf"        => $request->cpf,
            "celular"    => $request->celular,
            "nascimento" => Carbon::createFromFormat('d/m/Y', $request->nascimento)->format('Y-m-d'),
        ];

        if ($nomeFoto) {
            $dadosUsuario["foto"] = $nomeFoto;
        }

        $user->update($dadosUsuario);

        // 4. ATUALIZA INGRESSO
        $ingresso = Ingresso::where('pedido_id', $pedido->id)->first();

        $dadosIngresso = [
            'nome_completo'      => $user->name,
            'email'              => $user->email,
            'cpf'                => $user->cpf,
            'celular'            => $user->celular,
            'instituicao_ensino' => $request->instituicao_ensino,
            'curso'              => $request->curso,
            'nivel_ensino'       => $request->nivel_ensino,
        ];

        if ($nomeFoto) {
            $dadosIngresso['foto'] = env('BASE_STORAGE') . '/' . $nomeFoto;
        }

        $ingresso->update($dadosIngresso);

        return back()->with('success', 'Carteira editada com sucesso!');
    }




    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }


    public function print(Pedido $carteira)
    {
        if ($carteira->vigencia == '2025') {
            return view('admin.carteiras.print', ['carteira' => $carteira]);
        } elseif ($carteira->vigencia == '2026') {
            return view('admin.carteiras.carteira-2026', ['carteira' => $carteira]);
        }
    }
}
